/**
 * Helper functions to help write and run migrations
 *
 * Key design decisions:
 *
 * - all table names and column names will use camelCase
 * - all tables are strict, by default
 * - column PK will always be "id"
 * - all FK will always reference other tables via "id"
 * - there will always be createdAt/updatedAt
 * - FK constraints are always "ON DELETE CASCADE"
 */
import { BaseEntity } from './entity.js';
type SqliteTypes = 'text' | 'int' | 'float' | 'blob';
type SchemaValue = {
    pk: boolean;
    notNull: boolean;
    default?: string;
    type: SqliteTypes;
    fk: boolean;
    fkTo: string | null;
};
declare class ColumnInterface {
    ref: SchemaValue;
    constructor(type: SqliteTypes);
    notNull(): this;
    pk(): this;
    fk(to: string): this;
    default(val: string | number): this;
}
export type SchemaType = Record<string, ColumnInterface>;
/**
 * helper function to generate sql for "CREATE TABLE"
 *
 * NOTE: please read design assumptions in _migrator.ts
 * @param name of the table
 * @param schema of the table
 */
declare function createTable(name: string | typeof BaseEntity, schema: SchemaType): string;
declare function addColumn(tableName: string, columnName: string, dataType: SqliteTypes, notNull?: boolean, defaultValue?: string | number | null): string;
declare function removeColumn(tableName: string, columnName: string): string;
declare function dropTable(name: string): string;
declare const migrator: {
    text: () => ColumnInterface;
    int: () => ColumnInterface;
    float: () => ColumnInterface;
    blob: () => ColumnInterface;
};
export { migrator, createTable, dropTable, addColumn, removeColumn };
