import { DbErrorHandler } from './_base.repo.js';
import { DataSource } from '../dataSource.js';
import { Account, Profile } from '../_schema.js';
import { eq, gt } from '@dhaaga/orm';
import { AccountService } from './account.js';
import { ProfilePinnedTimelineService } from './profile-pinned-timeline.js';
import { ProfilePinnedTagService } from './profile-pinned-tag.js';
import { RandomUtil } from '@dhaaga/bridge';
@DbErrorHandler()
class Repo {
}
class Service {
    static getShownProfiles(db) {
        return db.profile.find({ active: true });
    }
    static renameProfileById(db, id, newName) {
        const profile = Service.getById(db, id);
        if (profile) {
            db.profile.updateById(profile.id, {
                name: newName,
            });
        }
    }
    static isDefaultProfile(db, profile) {
        return profile.uuid === 'DEFAULT';
    }
    static getById(db, id) {
        try {
            return db.profile.findOne({
                id: typeof id === 'string' ? parseInt(id) : id,
            });
        }
        catch (e) {
            return null;
        }
    }
    static getForAccount(db, acct) {
        return db.profile.find({ active: true, accountId: acct.id });
    }
    static getAllShown(db) {
        return db.profile.find({ active: true, visible: true });
    }
    static deselectAll(db) {
        db.profile.update({ id: gt(0) }, { selected: false });
    }
    static hideProfile(db, id) {
        db.profile.updateById(id, {
            visible: false,
        });
    }
    static unhideProfile(db, id) {
        db.profile.updateById(id, {
            visible: true,
        });
    }
    static removeProfile(db, id) {
        const match = db.profile.findOne({ id });
        if (match && match.uuid === 'DEFAULT')
            return;
        db.profile.updateById(id, {
            active: false,
        });
    }
    static getDefaultProfile(db, acct) {
        const match = db.profile.findOne({
            accountId: acct.id,
            uuid: 'DEFAULT',
        });
        if (match) {
            Service._postSelect(db, match);
            return match;
        }
        return null;
    }
    static getActiveProfile(db, acct) {
        let found = db.profile.findOne({
            accountId: acct.id,
            selected: true,
        });
        if (!found)
            found = Service.getDefaultProfile(db, acct);
        if (!found)
            throw new Error(`No profile found for account ${acct.identifier} (${acct.server})`);
        Service._postSelect(db, found);
        return found;
    }
    static setupDefaultProfile(db, acct) {
        let conflict = Service.getDefaultProfile(db, acct);
        if (!conflict) {
            db.profile.insert({
                accountId: acct.id,
                uuid: 'DEFAULT',
                name: 'Default',
                selected: true,
            });
        }
        else {
            conflict = Service.getDefaultProfile(db, acct);
        }
        if (conflict)
            Service._postInsert(db, conflict);
    }
    static selectDefaultProfile(db, acct) {
        const conflict = Service.getDefaultProfile(db, acct);
        if (conflict)
            db.profile.update({ id: eq(conflict.id) }, { selected: true });
    }
    static getOwnerAccount(db, profile) {
        return AccountService.getById(db, profile.accountId || -1);
    }
    static addProfile(db, acct, name) {
        const _key = RandomUtil.nanoId();
        db.profile.insert({
            accountId: acct.id,
            uuid: _key,
            name,
            selected: false,
        });
        const savedProfile = db.profile.findOne({
            uuid: _key,
        });
        if (savedProfile)
            Service._postInsert(db, savedProfile);
        return savedProfile;
    }
    static _postInsert(db, profile) {
        ProfilePinnedTimelineService.upsertDefaultTimelines(db, profile);
        ProfilePinnedTagService.upsertDefaultTags(db, profile);
    }
    static _postSelect(db, profile) {
        ProfilePinnedTimelineService.upsertDefaultTimelines(db, profile);
        ProfilePinnedTagService.upsertDefaultTags(db, profile);
    }
}
export { Repo as ProfileRepo, Service as ProfileService };
//# sourceMappingURL=profile.js.map