import { z } from 'zod';
import { Account } from '../_schema.js';
import { DbErrorHandler } from './_base.repo.js';
import { AccountMetadataService, } from './account-metadata.js';
import { KNOWN_SOFTWARE } from '@dhaaga/bridge';
import { DataSource } from '../dataSource.js';
import { gt } from '@dhaaga/orm';
import { RandomUtil } from '@dhaaga/bridge';
import { ProfileService } from './profile.js';
import { AccountCollectionService, ReservedCollection, } from './account-collection.js';
export const accountInsertDto = z.object({
    uuid: z.string(),
    identifier: z.string(),
    driver: z.string(),
    server: z.string(),
    username: z.string(),
    avatarUrl: z.string().nullable(),
    displayName: z.string().nullable(),
});
@DbErrorHandler()
export class Repo {
    static getById(db, id) {
        try {
            return db.account.findOne({
                id: Number(id),
            });
        }
        catch (e) {
            return null;
        }
    }
    static getByHandleFragments(db, server, username) {
        try {
            return db.account.findOne({
                server,
                username,
            });
        }
        catch (e) {
            return null;
        }
    }
    static upsert(db, dto) {
        const match = Repo.getByHandleFragments(db, dto.server, dto.username);
        if (match) {
            ProfileService.setupDefaultProfile(db, match);
            return match;
        }
        else {
            db.account.insert({
                uuid: RandomUtil.nanoId(),
                identifier: dto.identifier,
                driver: dto.driver,
                server: dto.server,
                username: dto.username,
                displayName: dto.displayName,
                avatarUrl: dto.avatarUrl,
            });
            const upserted = Repo.getByHandleFragments(db, dto.server, dto.username);
            ProfileService.setupDefaultProfile(db, upserted);
            return upserted;
        }
    }
    static updateSelectionFlag(db, id, flag) {
        db.account.update({ id }, { selected: flag });
    }
    static deselectAll(db) {
        db.account.update({ id: gt(0) }, {
            selected: false,
        });
        ProfileService.deselectAll(db);
    }
    static updateSoftware(db, id, driver) {
        db.account.update({ id }, { driver });
    }
    static getAll(db) {
        return db.account.find();
    }
    static removeById(db, id) {
        return db.db.runSync(`delete from account where id = ?`, id);
    }
    static getFirstSelected(db) {
        return db.account.findOne({ selected: true });
    }
}
class Service {
    static upsert(db, acct, metadata) {
        const upserted = Repo.upsert(db, acct);
        AccountMetadataService.upsertMultiple(db, upserted, metadata);
        ProfileService.setupDefaultProfile(db, upserted);
        return upserted;
    }
    static async selectSync(db, acct) { }
    static select(db, acct) {
        Repo.deselectAll(db);
        Repo.updateSelectionFlag(db, acct.id, true);
        ProfileService.selectDefaultProfile(db, acct);
    }
    static deselect(db, acct) {
        Repo.updateSelectionFlag(db, acct.id, false);
    }
    static getAll(db) {
        return Repo.getAll(db);
    }
    static getAllWithProfiles(db) {
        const accounts = Service.getAll(db);
        for (const account of accounts) {
            account.profiles = ProfileService.getForAccount(db, account);
        }
        return accounts;
    }
    static remove(db, acct) {
        return Repo.removeById(db, acct.id);
    }
    static removeById(db, id) {
        return Repo.removeById(db, id);
    }
    static updateDriver(db, acct, driver) {
        return Repo.updateSoftware(db, acct.id, driver.toString());
    }
    static getSelected(db) {
        const acct = Repo.getFirstSelected(db);
        if (!acct)
            throw new Error('No account selected!');
        Service._postSelect(db, acct);
        return acct;
    }
    static getById(db, id) {
        return Repo.getById(db, id);
    }
    static ensureAccountSelection(db) {
        try {
            const atLeastOne = db.account.find({ selected: true });
            if (atLeastOne.length === 0) {
                const match = db.account.findOne({ active: true });
                if (match) {
                    db.account.updateById(match.id, {
                        selected: true,
                    });
                }
                else {
                    console.log('[WARN]: no account to  default select');
                }
            }
        }
        catch (e) {
            console.log('[WARN]: failed to select default account', e);
        }
    }
    static _postSelect(db, acct) {
        AccountCollectionService.upsertReservedCollections(db, acct, ReservedCollection.DEFAULT);
    }
}
export { Service as AccountService, Repo as AccountRepo };
//# sourceMappingURL=account.js.map