import { jsx as _jsx } from "react/jsx-runtime";
import { RandomUtil } from '@dhaaga/bridge';
import { produce } from 'immer';
import { DataSource, ProfilePinnedTimelineService, ProfilePinnedUserService, ProfilePinnedTagService, APP_PINNED_OBJECT_TYPE, } from '@dhaaga/db';
import { createContext, useContext, useReducer, } from 'react';
import { timelineReducerBaseDefaults, } from './_timeline.shared.js';
var TimelineFetchMode;
(function (TimelineFetchMode) {
    TimelineFetchMode["IDLE"] = "Idle";
    TimelineFetchMode["HOME"] = "Home";
    TimelineFetchMode["LOCAL"] = "Local";
    TimelineFetchMode["FEDERATED"] = "Federated";
    TimelineFetchMode["SOCIAL"] = "Social";
    TimelineFetchMode["BUBBLE"] = "Bubble";
    TimelineFetchMode["HASHTAG"] = "Hashtag";
    TimelineFetchMode["USER"] = "User";
    TimelineFetchMode["LIST"] = "List";
    TimelineFetchMode["ANTENNA"] = "Antenna";
    TimelineFetchMode["REMOTE_TIMELINE"] = "Remote Timeline";
    TimelineFetchMode["ADD_NEW"] = "Add New";
    TimelineFetchMode["BOOKMARKS"] = "Bookmarks";
    TimelineFetchMode["LIKES"] = "Likes";
    TimelineFetchMode["FEED"] = "Feed";
})(TimelineFetchMode || (TimelineFetchMode = {}));
const DEFAULT = {
    ...timelineReducerBaseDefaults,
    feedType: TimelineFetchMode.IDLE,
    query: null,
    isWidgetVisible: false,
    items: [],
};
var ACTION;
(function (ACTION) {
    ACTION[ACTION["INIT"] = 0] = "INIT";
    ACTION[ACTION["RESET_USING_QUERY"] = 1] = "RESET_USING_QUERY";
    ACTION[ACTION["RESET_USING_PIN_ID"] = 2] = "RESET_USING_PIN_ID";
    ACTION[ACTION["APPEND_RESULTS"] = 3] = "APPEND_RESULTS";
    ACTION[ACTION["SET_QUERY_PARAMS"] = 4] = "SET_QUERY_PARAMS";
    ACTION[ACTION["SET_QUERY_OPTS"] = 5] = "SET_QUERY_OPTS";
    ACTION[ACTION["REQUEST_LOAD_MORE"] = 6] = "REQUEST_LOAD_MORE";
    ACTION[ACTION["RESET"] = 7] = "RESET";
    ACTION[ACTION["SHOW_WIDGET"] = 8] = "SHOW_WIDGET";
    ACTION[ACTION["HIDE_WIDGET"] = 9] = "HIDE_WIDGET";
    ACTION[ACTION["UPDATE_BOOKMARK_STATUS"] = 10] = "UPDATE_BOOKMARK_STATUS";
    ACTION[ACTION["UPDATE_BOOST_STATUS"] = 11] = "UPDATE_BOOST_STATUS";
    ACTION[ACTION["UPDATE_TRANSLATION_OUTPUT"] = 12] = "UPDATE_TRANSLATION_OUTPUT";
    ACTION[ACTION["UPDATE_LIKE_STATUS"] = 13] = "UPDATE_LIKE_STATUS";
    ACTION[ACTION["UPDATE_REACTION_STATE"] = 14] = "UPDATE_REACTION_STATE";
    ACTION[ACTION["POST_OBJECT_CHANGED"] = 15] = "POST_OBJECT_CHANGED";
    ACTION[ACTION["SETUP_USER_POST_TIMELINE"] = 16] = "SETUP_USER_POST_TIMELINE";
    ACTION[ACTION["SETUP_CUSTOM_FEED_TIMELINE"] = 17] = "SETUP_CUSTOM_FEED_TIMELINE";
})(ACTION || (ACTION = {}));
function reducer(state, action) {
    switch (action.type) {
        case ACTION.INIT: {
            return produce(state, (draft) => {
                draft.db = action.payload.db;
                draft.seen = new Set();
            });
        }
        case ACTION.RESET_USING_QUERY: {
            return produce(state, (draft) => {
                draft.feedType = action.payload.type;
                draft.sessionId = RandomUtil.nanoId();
                draft.appliedMaxId = null;
                draft.maxId = null;
                draft.minId = null;
                draft.seen = new Set();
            });
        }
        case ACTION.RESET_USING_PIN_ID: {
            const _id = action.payload.id;
            const _type = action.payload.type;
            switch (_type) {
                case 'feed': {
                    if (!state.db)
                        return state;
                    const match = ProfilePinnedTimelineService.findById(state.db, _id);
                    if (!match)
                        return state;
                    switch (match.category) {
                        case APP_PINNED_OBJECT_TYPE.AT_PROTO_MICROBLOG_HOME:
                        case APP_PINNED_OBJECT_TYPE.AP_PROTO_MICROBLOG_HOME: {
                            return produce(state, (draft) => {
                                draft.feedType = TimelineFetchMode.HOME;
                                draft.sessionId = RandomUtil.nanoId();
                                draft.seen = new Set();
                            });
                        }
                        case APP_PINNED_OBJECT_TYPE.AT_PROTO_MICROBLOG_FEED: {
                            return produce(state, (draft) => {
                                draft.feedType = TimelineFetchMode.FEED;
                                draft.sessionId = RandomUtil.nanoId();
                                draft.seen = new Set();
                                draft.query = {
                                    id: match.uri,
                                    label: match.displayName,
                                };
                            });
                        }
                        case APP_PINNED_OBJECT_TYPE.AP_PROTO_MICROBLOG_LOCAL: {
                            return produce(state, (draft) => {
                                draft.feedType = TimelineFetchMode.LOCAL;
                                draft.sessionId = RandomUtil.nanoId();
                                draft.seen = new Set();
                            });
                        }
                        case APP_PINNED_OBJECT_TYPE.AP_PROTO_MICROBLOG_SOCIAL: {
                            return produce(state, (draft) => {
                                draft.feedType = TimelineFetchMode.SOCIAL;
                                draft.sessionId = RandomUtil.nanoId();
                                draft.seen = new Set();
                            });
                        }
                        case APP_PINNED_OBJECT_TYPE.AP_PROTO_MICROBLOG_BUBBLE: {
                            return produce(state, (draft) => {
                                draft.feedType = TimelineFetchMode.BUBBLE;
                                draft.sessionId = RandomUtil.nanoId();
                                draft.seen = new Set();
                            });
                        }
                        case APP_PINNED_OBJECT_TYPE.AP_PROTO_MICROBLOG_GLOBAL: {
                            return produce(state, (draft) => {
                                draft.feedType = TimelineFetchMode.FEDERATED;
                                draft.sessionId = RandomUtil.nanoId();
                                draft.seen = new Set();
                            });
                        }
                        default: {
                            console.log('[WARN]: tried to initialise with an incompatible timeline', _id, _type);
                            return state;
                        }
                    }
                }
                case 'user': {
                    if (!state.db)
                        return state;
                    const match = ProfilePinnedUserService.findById(state.db, _id);
                    if (!match)
                        return state;
                    switch (match.category) {
                        case APP_PINNED_OBJECT_TYPE.AP_PROTO_MICROBLOG_USER_LOCAL:
                        case APP_PINNED_OBJECT_TYPE.AP_PROTO_MICROBLOG_USER_REMOTE: {
                            return produce(state, (draft) => {
                                draft.feedType = TimelineFetchMode.USER;
                                draft.sessionId = RandomUtil.nanoId();
                                draft.query = {
                                    id: match.identifier,
                                    label: match.displayName || match.username,
                                };
                                draft.seen = new Set();
                            });
                        }
                        default: {
                            console.log('[WARN]: tried to initialise with an incompatible user target', _id, _type);
                            return state;
                        }
                    }
                }
                case 'tag': {
                    if (!state.db)
                        return state;
                    const match = ProfilePinnedTagService.findById(state.db, _id);
                    if (!match)
                        return state;
                    switch (match.category) {
                        case APP_PINNED_OBJECT_TYPE.AP_PROTO_MICROBLOG_TAG_LOCAL:
                        case APP_PINNED_OBJECT_TYPE.AP_PROTO_MICROBLOG_TAG_REMOTE: {
                            return produce(state, (draft) => {
                                draft.feedType = TimelineFetchMode.HASHTAG;
                                draft.query = {
                                    id: match.name,
                                    label: match.name,
                                };
                                draft.seen = new Set();
                            });
                        }
                        default: {
                            console.log('[WARN]: tried to initialise with an incompatible tag target', _id, _type);
                            return state;
                        }
                    }
                }
                default: {
                    console.log('[WARN]: app does not recognise this query', _id, _type);
                    return state;
                }
            }
        }
        case ACTION.APPEND_RESULTS: {
            const copy = Array.from(state.items);
            return produce(state, (draft) => {
                for (const item of action.payload.items) {
                    if (state.seen.has(item.id))
                        continue;
                    copy.push(item);
                    draft.seen.add(item.id);
                }
                draft.items = copy;
                draft.maxId = action.payload.maxId;
            });
        }
        case ACTION.SET_QUERY_OPTS: {
            return produce(state, (draft) => {
                draft.maxId = null;
                draft.minId = null;
                draft.items = [];
                draft.isEol = false;
                draft.isFirstLoad = false;
                draft.isWidgetVisible = false;
                draft.opts = action.payload;
            });
        }
        case ACTION.REQUEST_LOAD_MORE: {
            return produce(state, (draft) => {
                draft.appliedMaxId = state.maxId;
            });
        }
        case ACTION.RESET: {
            return produce(state, (draft) => {
                draft.items = [];
                draft.maxId = null;
                draft.minId = null;
                draft.isEol = false;
                draft.appliedMaxId = null;
                draft.isFirstLoad = false;
                draft.isWidgetVisible = false;
                draft.seen = new Set();
            });
        }
        case ACTION.SHOW_WIDGET: {
            return produce(state, (draft) => {
                draft.isWidgetVisible = true;
            });
        }
        case ACTION.HIDE_WIDGET: {
            return produce(state, (draft) => {
                draft.isWidgetVisible = true;
            });
        }
        case ACTION.UPDATE_BOOKMARK_STATUS: {
            const _id = action.payload.id;
            const _value = action.payload.value;
            if (_id === undefined || _value === undefined)
                return state;
            return produce(state, (draft) => {
                for (const post of draft.items) {
                    if (post.id === _id) {
                        post.interaction.bookmarked = _value;
                        post.state.isBookmarkStateFinal = true;
                    }
                    if (post.boostedFrom?.id === _id) {
                        post.boostedFrom.interaction.bookmarked = _value;
                        post.boostedFrom.state.isBookmarkStateFinal = true;
                    }
                }
            });
        }
        case ACTION.UPDATE_BOOST_STATUS: {
            const _id = action.payload.id;
            const _delta = action.payload.delta;
            if (!_id)
                return state;
            return produce(state, (draft) => {
                for (const post of draft.items) {
                    if (post.id === _id) {
                        post.interaction.boosted = _delta != -1;
                        post.stats.boostCount += _delta;
                    }
                    if (post.boostedFrom?.id === _id) {
                        post.boostedFrom.interaction.boosted = _delta != -1;
                        post.stats.boostCount += _delta;
                    }
                }
            });
        }
        case ACTION.UPDATE_TRANSLATION_OUTPUT: {
            const _id = action.payload.id;
            const _outputText = action.payload.outputText;
            const _outputType = action.payload.outputType;
            if (_id === undefined ||
                _outputText === undefined ||
                _outputType === undefined)
                return state;
            return produce(state, (draft) => {
                for (const post of draft.items) {
                    if (post.id === _id) {
                        post.calculated.translationOutput = _outputText;
                        post.calculated.translationType = _outputType;
                    }
                    if (post.boostedFrom?.id === _id) {
                        post.boostedFrom.calculated.translationOutput = _outputText;
                        post.boostedFrom.calculated.translationType = _outputType;
                    }
                }
            });
        }
        case ACTION.UPDATE_LIKE_STATUS: {
            const _id = action.payload.id;
            const _delta = action.payload.delta;
            if (_id === undefined || _delta === undefined)
                return state;
            return produce(state, (draft) => {
                for (const post of draft.items) {
                    if (post.id === _id) {
                        post.interaction.liked = _delta != -1;
                        post.stats.likeCount += _delta;
                    }
                    if (post.boostedFrom?.id === _id) {
                        post.boostedFrom.interaction.liked = _delta != -1;
                        post.boostedFrom.stats.likeCount += _delta;
                    }
                }
            });
        }
        case ACTION.UPDATE_REACTION_STATE: {
            const _id = action.payload.id;
            const _state = action.payload.state;
            return produce(state, (draft) => {
                for (const post of draft.items) {
                    if (post.id === _id)
                        post.stats.reactions = _state;
                    if (post.boostedFrom?.id === _id)
                        post.boostedFrom.stats.reactions = _state;
                }
            });
        }
        case ACTION.POST_OBJECT_CHANGED: {
            const newItem = action.payload.item;
            if (!newItem)
                return state;
            return produce(state, (draft) => {
                draft.items = draft.items.map((post) => post.id === newItem.id ? newItem : post);
            });
        }
        case ACTION.SETUP_USER_POST_TIMELINE: {
            return produce(state, (draft) => {
                draft.feedType = TimelineFetchMode.USER;
                draft.sessionId = RandomUtil.nanoId();
                draft.query = {
                    id: action.payload.id,
                    label: action.payload.label,
                };
                draft.seen = new Set();
            });
        }
        case ACTION.SETUP_CUSTOM_FEED_TIMELINE: {
            return produce(state, (draft) => {
                draft.feedType = TimelineFetchMode.FEED;
                draft.sessionId = RandomUtil.nanoId();
                draft.query = {
                    id: action.payload.uri,
                    label: action.payload.label,
                };
            });
        }
        default:
            return state;
    }
}
const StateCtx = createContext(null);
const DispatchCtx = createContext(null);
const usePostTimelineState = () => useContext(StateCtx);
const usePostTimelineDispatch = () => useContext(DispatchCtx);
function Ctx({ children }) {
    const [state, dispatch] = useReducer(reducer, DEFAULT);
    return (_jsx(StateCtx.Provider, { value: state, children: _jsx(DispatchCtx.Provider, { value: dispatch, children: children }) }));
}
export { TimelineFetchMode, Ctx as PostTimelineCtx, usePostTimelineState, usePostTimelineDispatch, ACTION as PostTimelineStateAction, };
//# sourceMappingURL=post-timeline.js.map