import { BaseApiAdapter, } from '../adapters/index.js';
import { DriverService } from './driver.js';
import { RandomUtil } from '../utils/index.js';
var KNOWN_SOFTWARE;
(function (KNOWN_SOFTWARE) {
    KNOWN_SOFTWARE["AKKOMA"] = "akkoma";
    KNOWN_SOFTWARE["BLUESKY"] = "bluesky";
    KNOWN_SOFTWARE["CHERRYPICK"] = "cherrypick";
    KNOWN_SOFTWARE["FIREFISH"] = "firefish";
    KNOWN_SOFTWARE["FRIENDICA"] = "friendica";
    KNOWN_SOFTWARE["GOTOSOCIAL"] = "gotosocial";
    KNOWN_SOFTWARE["HOMETOWN"] = "hometown";
    KNOWN_SOFTWARE["ICESHRIMP"] = "iceshrimp";
    KNOWN_SOFTWARE["KMYBLUE"] = "kmyblue";
    KNOWN_SOFTWARE["LEMMY"] = "lemmy";
    KNOWN_SOFTWARE["MASTODON"] = "mastodon";
    KNOWN_SOFTWARE["MEISSKEY"] = "meisskey";
    KNOWN_SOFTWARE["MISSKEY"] = "misskey";
    KNOWN_SOFTWARE["PEERTUBE"] = "peertube";
    KNOWN_SOFTWARE["PIXELFED"] = "pixelfed";
    KNOWN_SOFTWARE["PLEROMA"] = "pleroma";
    KNOWN_SOFTWARE["SHARKEY"] = "sharkey";
    KNOWN_SOFTWARE["UNKNOWN"] = "unknown";
})(KNOWN_SOFTWARE || (KNOWN_SOFTWARE = {}));
class ActivityPubService {
    static canBookmark(driver) {
        return DriverService.canBookmark(driver);
    }
    static canLike(driver) {
        return DriverService.canLike(driver);
    }
    static canAddReactions(driver) {
        DriverService.canReact(driver);
    }
    static mastodonLike(driver) {
        return [
            KNOWN_SOFTWARE.MASTODON,
            KNOWN_SOFTWARE.PLEROMA,
            KNOWN_SOFTWARE.AKKOMA,
        ].includes(driver);
    }
    static supportsV2(driver) {
        return DriverService.supportsMastoApiV2(driver);
    }
    static supportsQuotesNatively(driver) {
        return DriverService.canQuote(driver);
    }
    static pleromaLike(driver) {
        return [KNOWN_SOFTWARE.PLEROMA, KNOWN_SOFTWARE.AKKOMA].includes(driver);
    }
    static misskeyLike(driver) {
        return DriverService.supportsMisskeyApi(driver);
    }
    static blueskyLike(driver) {
        return DriverService.supportsAtProto(driver);
    }
    static async toggleBookmark(client, id, localState) {
        return localState
            ? client.statuses.unBookmark(id)
            : client.statuses.bookmark(id);
    }
    static async toggleLike(client, localState, idA, idB) {
        if (localState)
            return client.statuses.removeLike(idA, idB);
        return client.statuses.like(idA, idB);
    }
    static async toggleBoost(client, id, localState, domain) {
        if (ActivityPubService.misskeyLike(domain)) {
            if (localState) {
                const { error } = await client.statuses.unrenote(id);
                if (error) {
                    console.log('[WARN]: failed to remove boost', error);
                    return null;
                }
                return -1;
            }
            else {
                const { error } = await client.statuses.renote({
                    renoteId: id,
                    visibility: 'followers',
                    localOnly: true,
                });
                if (error) {
                    console.log('[WARN]: failed to boost', error);
                    return null;
                }
                return +1;
            }
        }
        else if (domain === KNOWN_SOFTWARE.MASTODON) {
            if (localState) {
                const { error } = await client.statuses.removeBoost(id);
                if (error) {
                    console.log('[WARN]: failed to remove boost', error);
                    return null;
                }
                return -1;
            }
            else {
                const { error } = await client.statuses.boost(id);
                if (error) {
                    console.log('[WARN]: failed to boost', error);
                    return null;
                }
                return 1;
            }
        }
        else {
            if (localState) {
                const { error } = await client.statuses.removeBoost(id);
                if (error) {
                    console.log('[WARN]: failed to remove boost', error);
                    return null;
                }
                return -1;
            }
            else {
                const { error } = await client.statuses.boost(id);
                if (error) {
                    console.log('[WARN]: failed to boost', error);
                    return null;
                }
                return 1;
            }
        }
    }
    static async detectSoftware(urlLike) {
        const client = new BaseApiAdapter();
        const { data, error } = await client.instances.getSoftwareInfo(urlLike);
        if (error || !data)
            return null;
        return data.software;
    }
    static async signInUrl(urlLike, token) {
        const client = new BaseApiAdapter();
        return client.instances.getLoginUrl(urlLike, {
            appCallback: 'https://suvam.io',
            appName: 'Dhaaga',
            appClientId: token?.clientId,
            appClientSecret: token?.clientSecret,
            uuid: RandomUtil.nanoId(),
        });
    }
    static async getBookmarkState(client, id) {
        const { data, error } = await client.statuses.getState(id);
        if (error) {
            console.log('[WARN]: error fetching bookmarked state');
            return null;
        }
        return data.isFavorited;
    }
}
export default ActivityPubService;
//# sourceMappingURL=activitypub.service.js.map