import ActivityPubService from './activitypub.service.js';
import { z } from 'zod';
import activitypubService from './activitypub.service.js';
import { PostParser } from '../parsers/post.js';
const MISSKEY_LOCAL_EX = /:(.*?):/;
const MISSKEY_LOCAL_ALT_EX = /:(.*?)@.:/;
const MISSKEY_REMOTE_EX = /:(.*?)@(.*?):/;
const PLEROMA_REMOTE_EX = /(.*?)@(.*?)/;
const pleromaReactionItemSchema = z.object({
    count: z.number().positive(),
    me: z.boolean(),
    url: z.string(),
    name: z.string(), // pleroma, resolved into id by parser
    accountIds: z.array(z.string()),
});
const activityPubReactionItemSchema = z.object({
    id: z.string(),
    count: z.number().positive(),
    me: z.boolean(),
    accounts: z.array(z.string()),
    url: z.string().nullable().optional(),
});
const ActivityPubReactionStateSchema = z.array(activityPubReactionItemSchema);
class ActivityPubReactionsService {
    static extractReactionCode(input, domain, subdomain) {
        if (activitypubService.misskeyLike(domain)) {
            return { id: input };
        }
        if (MISSKEY_LOCAL_ALT_EX.test(input)) {
            const _name = MISSKEY_LOCAL_ALT_EX.exec(input)[1];
            return { id: _name };
        }
        if (MISSKEY_LOCAL_EX.test(input)) {
            const _name = MISSKEY_LOCAL_EX.exec(input)[1];
            return { id: _name };
        }
        return { id: input };
    }
    static cannotReact(id) {
        if (!id)
            return false;
        return ((MISSKEY_REMOTE_EX.test(id) || PLEROMA_REMOTE_EX.test(id)) &&
            !MISSKEY_LOCAL_ALT_EX.test(id));
    }
    static renderData(input, { me, calculated, cache, }) {
        let retval = [];
        for (const reaction of input) {
            let IS_ME = reaction.me;
            if (reaction.accounts && reaction.accounts.includes(me)) {
                IS_ME = true;
            }
            const IS_REMOTE = (MISSKEY_REMOTE_EX.test(reaction.id) ||
                PLEROMA_REMOTE_EX.test(reaction.id)) &&
                !MISSKEY_LOCAL_ALT_EX.test(reaction.id);
            const BASE = {
                name: reaction.id,
                count: reaction.count,
                interactable: !IS_REMOTE,
                me: IS_ME,
            };
            if (MISSKEY_LOCAL_ALT_EX.test(reaction.id)) {
                const _name = MISSKEY_LOCAL_ALT_EX.exec(reaction.id)[1];
                const match = cache.find((o) => o.shortCode === _name || o.aliases.includes(_name));
                if (!match) {
                    console.log('[WARN]: local emoji not found for', _name);
                    continue;
                }
                retval.push({
                    ...BASE,
                    type: 'image',
                    url: match.url,
                });
            }
            else if (MISSKEY_REMOTE_EX.test(reaction.id)) {
                const _name = MISSKEY_LOCAL_EX.exec(reaction.id)[1];
                const match = calculated.find((o) => o.name === _name);
                if (!match) {
                    console.log('[WARN]: failed to resolve remote misskey reaction', reaction, calculated);
                    continue;
                }
                retval.push({
                    ...BASE,
                    type: 'image',
                    url: match.url,
                    width: match.width,
                    height: match.height,
                });
            }
            else if (MISSKEY_LOCAL_EX.test(reaction.id)) {
                const _name = MISSKEY_LOCAL_EX.exec(reaction.id)[1];
                const match = calculated.find((o) => o.name === _name);
                const cacheHit = cache.find((o) => o.shortCode === _name || o.aliases.includes(_name));
                if (!match?.url && !cacheHit) {
                    console.log('[WARN]: failed to resolve local misskey emoji');
                    continue;
                }
                retval.push({
                    ...BASE,
                    type: 'image',
                    url: match?.url || cacheHit?.url,
                    width: match?.width,
                    height: match?.height,
                });
            }
            else if (PLEROMA_REMOTE_EX.test(reaction.id)) {
                const match = calculated.find((o) => o.name === reaction.id);
                if (!match && !reaction.url) {
                    console.log('[WARN]: failed to resolve remote pleroma reaction');
                    continue;
                }
                retval.push({
                    ...BASE,
                    type: 'image',
                    url: reaction.url || match?.url,
                    width: match?.width,
                    height: match?.height,
                });
            }
            else {
                const cacheHit = cache.find((o) => o.shortCode === reaction.id || o.aliases.includes(reaction.id));
                if (cacheHit || reaction.url) {
                    retval.push({
                        name: reaction.id,
                        count: reaction.count,
                        type: 'image',
                        url: reaction.url || cacheHit?.url,
                        interactable: true,
                        me: reaction.me,
                    });
                }
                else {
                    retval.push({
                        name: reaction.id,
                        count: reaction.count,
                        type: 'text',
                        interactable: true,
                        me: reaction.me,
                    });
                }
            }
        }
        return retval;
    }
    static async removeReaction(client, postId, reactionId, domain, setLoading) {
        setLoading(true);
        if (ActivityPubService.pleromaLike(domain)) {
            const { data, error } = await client.statuses.removeReaction(postId, reactionId);
            if (error) {
                console.log('[WARN]: failed to add reaction', error);
                return null;
            }
            setLoading(false);
            return data.emojiReactions.map((o) => ({
                id: o.name.length > 2 ? `:${o.name}:` : o.name,
                me: o.me,
                count: o.count,
                accounts: o.accountIds || [],
            }));
        }
        else if (ActivityPubService.misskeyLike(domain)) {
            const { error } = await client.statuses.removeReaction(postId, reactionId);
            if (error && error.code) {
                console.log('[WARN]: failed to remove reaction', error);
                return null;
            }
            return ActivityPubReactionsService.syncMisskeyReactionState(client, postId, domain, setLoading);
        }
        return null;
    }
    static async syncMisskeyReactionState(client, postId, domain, setLoading) {
        const { data: newStateData, error: newStateError } = await client.statuses.get(postId);
        if (newStateError) {
            setLoading(false);
            return [];
        }
        const status = PostParser.rawToInterface(newStateData, domain);
        setLoading(false);
        return status.getReactions(status.getMyReaction() || 'N/A');
    }
    static async addReaction(client, postId, reactionId, domain, setLoading) {
        setLoading(true);
        if (ActivityPubService.pleromaLike(domain)) {
            const { data, error } = await client.statuses.addReaction(postId, reactionId);
            if (error) {
                console.log('[WARN]: failed to add reaction', error);
                return [];
            }
            setLoading(false);
            return data.emojiReactions.map((o) => ({
                id: o.name.length > 2 ? `:${o.name}:` : o.name,
                me: o.me,
                count: o.count,
                accounts: o.accountIds || [],
            }));
        }
        else if (ActivityPubService.misskeyLike(domain)) {
            if (!MISSKEY_LOCAL_EX.test(reactionId)) {
                reactionId = `:${reactionId}:`;
            }
            const { error } = await client.statuses.addReaction(postId, reactionId);
            if (error) {
                console.log('[WARN]: failed to add reaction', error);
                return [];
            }
            return ActivityPubReactionsService.syncMisskeyReactionState(client, postId, domain, setLoading);
        }
        return [];
    }
}
export { activityPubReactionItemSchema, ActivityPubReactionStateSchema };
export default ActivityPubReactionsService;
//# sourceMappingURL=activitypub-reactions.service.js.map