import { z } from 'zod';
import { TextNodeParser } from './text-nodes.js';
import { ActivityPubUserAdapter, } from '../implementors/profile/_interface.js';
import { ActivitypubHelper } from '../index.js';
export const appUserObjectSchema = z.object({
    id: z.string(),
    avatarUrl: z.string(),
    displayName: z.string().nullable(),
    parsedDisplayName: z.array(z.any()),
    handle: z.string(),
    instance: z.string(),
    banner: z.string().nullable().optional(),
    meta: z.object({
        isProfileLocked: z.boolean(),
        isBot: z.boolean(),
        fields: z.array(z.object({
            name: z.string().optional(),
            value: z.string().optional(),
            verifiedAt: z.string().nullable().optional(),
            cid: z.string().optional(),
            cts: z.string().optional(),
            src: z.string().optional(),
            uri: z.string().optional(),
            val: z.string().optional(),
        })),
    }),
    description: z.string(),
    parsedDescription: z.array(z.any()),
    stats: z.object({
        posts: z.number().nullable(),
        followers: z.number().nullable(),
        following: z.number().nullable(),
    }),
    calculated: z.object({
        emojis: z.map(z.string(), z.string()),
    }),
    relationship: z.object({
        blocking: z.boolean().nullable(),
        blockedBy: z.boolean().nullable(),
        domainBlocking: z.boolean().nullable(),
        followedBy: z.boolean().nullable(),
        following: z.boolean().nullable(),
        muting: z.boolean().nullable(),
        mutingNotifications: z.boolean().nullable(),
        note: z.string().nullable(),
        requested: z.boolean().nullable(),
        requestedBy: z.boolean().nullable(),
        showingReblogs: z.boolean().nullable(),
    }),
});
export const APP_USER_DEFAULT_RELATIONSHIP = {
    blocking: null,
    blockedBy: null,
    domainBlocking: null,
    followedBy: null,
    following: null,
    muting: null,
    mutingNotifications: null,
    note: null,
    requested: null,
    requestedBy: null,
    showingReblogs: null,
};
class Parser {
    static rawToInterface(input, driver) {
        if (Array.isArray(input)) {
            return input
                .filter((o) => !!o)
                .map((o) => ActivityPubUserAdapter(o, driver));
        }
        else {
            return ActivityPubUserAdapter(input, driver);
        }
    }
    static interfaceToJson(input, { driver, server, }) {
        if (!input || !input.getId())
            return null;
        const dto = {
            id: input.getId(),
            displayName: input.getDisplayName() || '',
            parsedDisplayName: TextNodeParser.parse(driver, input.getDisplayName() || ''),
            description: input.getDescription() || '',
            parsedDescription: TextNodeParser.parse(driver, input.getDescription() || ''),
            avatarUrl: input.getAvatarUrl() || '',
            banner: input.getBannerUrl(),
            handle: ActivitypubHelper.getHandle(input?.getAccountUrl(server), server, driver),
            instance: input.getInstanceUrl() || server,
            stats: {
                posts: input.getPostCount() || 0,
                followers: input.getFollowersCount() || 0,
                following: input.getFollowingCount() || 0,
            },
            meta: {
                isBot: input.getIsBot() || false,
                isProfileLocked: input.getIsLockedProfile() || false,
                fields: input.getFields() || [],
            },
            calculated: {
                emojis: input.getEmojiMap(),
            },
            relationship: APP_USER_DEFAULT_RELATIONSHIP,
        };
        const { data, error, success } = appUserObjectSchema.safeParse(dto);
        if (!success) {
            console.log('[ERROR]: user dto validation failed', error);
            return null;
        }
        return data;
    }
    static parse(input, driver, server) {
        if (Array.isArray(input)) {
            return input
                .map((o) => Parser.rawToInterface(o, driver))
                .map((o) => Parser.interfaceToJson(o, {
                driver,
                server,
            }))
                .filter((o) => !!o);
        }
        else {
            return Parser.interfaceToJson(Parser.rawToInterface(input, driver), {
                driver,
                server,
            });
        }
    }
}
export { Parser as UserParser };
//# sourceMappingURL=user.js.map