import { z } from 'zod';
import { UserParser } from './user.js';
import { TextNodeParser } from './text-nodes.js';
import { DriverService } from '../services/driver.js';
import { Err, Ok, RandomUtil } from '../utils/index.js';
import { ApiErrorCode } from '../types/result.types.js';
import { ActivitypubStatusAdapter } from '../implementors/status/_adapters.js';
import { ActivitypubHelper } from '../index.js';
const mentionObjectSchema = z.object({
    id: z.string(),
    handle: z.string().optional(),
    url: z.string().optional(),
    acct: z.string().optional().nullable(),
    username: z.string().optional().nullable(),
});
const ActivityPubReactionStateSchema = z.array(z.object({
    id: z.string(),
    count: z.number().positive(),
    me: z.boolean(),
    accounts: z.array(z.string()),
    url: z.string().nullable().optional(),
}));
const ActivityPubBoostedByDto = z.object({
    id: z.string(),
    avatarUrl: z.string(),
    displayName: z.string().nullable().optional(),
    parsedDisplayName: z.array(z.any()),
    handle: z.string().regex(/^@.*?@?.*?$/),
    instance: z.string(),
});
const AppActivityPubMediaDto = z.object({
    url: z.string(),
    previewUrl: z.string().nullable().optional(),
    width: z.number().optional().nullable(), // bsky can be null
    height: z.number().optional().nullable(), // bsky can be null
    alt: z.string().nullable(),
    type: z.string(),
    blurhash: z.string().nullable(),
});
const AppPostStatsDto = z.object({
    replyCount: z.number().nonnegative(),
    boostCount: z.number().nonnegative(),
    likeCount: z.number().nonnegative(),
    reactions: ActivityPubReactionStateSchema,
});
const ActivityPubStatusItemDto = z.object({
    uuid: z.string(),
    id: z.string(),
    visibility: z.string(),
    createdAt: z.string(),
    postedBy: ActivityPubBoostedByDto,
    content: z.object({
        raw: z.string().nullable().optional(),
        parsed: z.array(z.any()),
        media: z.array(AppActivityPubMediaDto),
    }),
    interaction: z.object({
        boosted: z.boolean(),
        liked: z.boolean(),
        bookmarked: z.boolean(),
    }),
    stats: AppPostStatsDto,
    calculated: z.object({
        mediaContainerHeight: z.number(),
        emojis: z.map(z.string(), z.string()),
        translationOutput: z.string().optional(),
        translationType: z.string().optional(),
        reactionEmojis: z.array(z.object({
            height: z.number().nullable().optional(),
            width: z.number().nullable().optional(),
            name: z.string(),
            url: z.string().url(),
        })),
        mentions: z.array(z.object({
            id: z.string().optional().nullable(),
            text: z.string().optional(),
            url: z.string().nullable().optional(),
            username: z.string().optional().nullable(),
            acct: z.string().optional().nullable(),
        })),
    }),
    meta: z.object({
        sensitive: z.boolean(),
        cw: z.string().nullable(),
        isBoost: z.boolean(),
        isReply: z.boolean(),
        mentions: z.array(z.object({
            id: z.string(), // lazy loaded for misskey forks
            handle: z.string().optional(),
            url: z.string().optional(),
            acct: z.string().optional().nullable(),
            username: z.string().optional().nullable(),
        })),
        cid: z.string().nullable().optional(),
        uri: z.string().nullable().optional(),
    }),
    state: z.object({
        isBookmarkStateFinal: z.boolean(),
    }),
    atProto: z
        .object({
        viewer: z
            .object({
            like: z.string().nullable().optional(),
            embeddingDisabled: z.boolean().optional(),
            pinned: z.any().optional(),
            repost: z.any().optional(),
            replyDisabled: z.boolean().optional(),
            threadMuted: z.boolean().optional(),
        })
            .optional(),
    })
        .nullable()
        .optional(),
});
const ActivityPubStatusLevelTwo = ActivityPubStatusItemDto.extend({
    replyTo: ActivityPubStatusItemDto.nullable().optional(), // Misskey/Firefish natively supports quote boosting
    boostedFrom: ActivityPubStatusItemDto.nullable().optional(), // Pleroma feature
    quotedFrom: ActivityPubStatusItemDto.nullable().optional(),
});
const postObjectSchema = ActivityPubStatusLevelTwo.extend({
    replyTo: ActivityPubStatusLevelTwo.nullable().optional(), // Misskey/Firefish natively supports quote boosting
    boostedFrom: ActivityPubStatusLevelTwo.nullable().optional(), // Pleroma feature
    quotedFrom: ActivityPubStatusLevelTwo.nullable().optional(), // Bluesky feature
    rootPost: ActivityPubStatusItemDto.nullable().optional(),
});
class Parser {
    static export(input, domain, subdomain) {
        if (!input)
            return null;
        const IS_BOOKMARK_RESOLVED = DriverService.supportsMastoApiV1(domain);
        const IS_ATPROTO = DriverService.supportsAtProto(domain);
        const medias = input?.getMediaAttachments();
        const user = UserParser.rawToInterface(input.getUser(), domain);
        let handle = IS_ATPROTO
            ? `@${user.getUsername()}`
            : ActivitypubHelper.getHandle(input.getAccountUrl(subdomain) || '', subdomain);
        const parsedContent = TextNodeParser.parse(domain, input.getContent() || '', input.getFacets());
        const parsedDisplayName = IS_ATPROTO
            ? [
                {
                    uuid: RandomUtil.nanoId(),
                    type: 'para',
                    nodes: [
                        {
                            uuid: RandomUtil.nanoId(),
                            type: 'text',
                            text: user.getDisplayName(),
                        },
                    ],
                },
            ]
            : TextNodeParser.parse(domain, input.getDisplayName() || '', input.getFacets());
        return {
            uuid: RandomUtil.nanoId(),
            id: input.getId(),
            visibility: input.getVisibility(),
            createdAt: input.getCreatedAt(),
            postedBy: {
                id: user.getId(),
                avatarUrl: user.getAvatarUrl() || '',
                displayName: user.getDisplayName(),
                parsedDisplayName: parsedDisplayName || [],
                handle: handle,
                instance: user.getInstanceUrl() || subdomain,
            },
            content: {
                raw: input.getContent(),
                parsed: parsedContent || [],
                media: medias.map((o) => ({
                    height: o.getHeight(),
                    width: o.getWidth(),
                    alt: o.getAltText() || null,
                    blurhash: o.getBlurHash() || null,
                    type: o.getType(),
                    url: o.getUrl(),
                    previewUrl: o.getPreviewUrl(),
                })) || [],
            },
            stats: {
                replyCount: input.getRepliesCount(),
                boostCount: input.getRepostsCount(),
                likeCount: input.getFavouritesCount(),
                reactions: input.getReactions(input.getMyReaction() || ''),
            },
            interaction: {
                bookmarked: input.getIsBookmarked() || false,
                boosted: input.getIsRebloggedByMe() || false,
                liked: input.getIsFavourited() || false,
            },
            calculated: {
                emojis: new Map([...user.getEmojiMap(), ...input.getCachedEmojis()]),
                mediaContainerHeight: 0, // height,
                reactionEmojis: input.getReactionEmojis(),
                mentions: input.getMentions(),
            },
            meta: {
                sensitive: input.getIsSensitive(),
                cw: input.getSpoilerText() || null,
                isBoost: input.isReposted(),
                isReply: input.isReply(),
                mentions: input.getMentions(),
                cid: input.getCid(),
                uri: input.getUri(),
            },
            state: {
                isBookmarkStateFinal: IS_BOOKMARK_RESOLVED,
            },
            atProto: {
                viewer: DriverService.supportsAtProto(domain)
                    ? input.getViewer()
                    : undefined,
            },
        };
    }
    static rawToInterface(input, driver) {
        if (Array.isArray(input)) {
            return input
                .filter((o) => !!o)
                .map((o) => ActivitypubStatusAdapter(o, driver));
        }
        else {
            return ActivitypubStatusAdapter(input, driver);
        }
    }
    static interfaceToJson(input, { driver, server, }) {
        if (!input)
            return null;
        const IS_SHARE = input.isReposted();
        const HAS_PARENT = input.isReply();
        const HAS_ROOT = input.hasRootAvailable();
        let sharedFrom = IS_SHARE
            ? Parser.parse(input.getRepostedStatusRaw(), driver, server)
            : null;
        let replyTo = HAS_PARENT
            ? Parser.parse(input.getParentRaw(), driver, server)
            : null;
        let root = HAS_ROOT
            ? Parser.parse(input.getRootRaw(), driver, server)
            : null;
        const dto = HAS_PARENT &&
            (DriverService.supportsAtProto(driver) ||
                DriverService.supportsMisskeyApi(driver))
            ?
                {
                    ...Parser.export(input, driver, server),
                    boostedFrom: sharedFrom,
                    replyTo,
                    rootPost: root,
                }
            : {
                ...Parser.export(input, driver, server),
                boostedFrom: sharedFrom,
            };
        const { data, error, success } = postObjectSchema.safeParse(dto);
        if (!success) {
            console.log('[ERROR]: status item dto validation failed', error);
            console.log('[INFO]: generated object', dto);
            input.print();
            return null;
        }
        return data;
    }
    static parse(input, driver, server) {
        if (input instanceof Array) {
            return input
                .map((o) => Parser.rawToInterface(o, driver))
                .filter((o) => !!o)
                .map((o) => Parser.interfaceToJson(o, {
                driver,
                server,
            }))
                .filter((o) => !!o);
        }
        else {
            try {
                if (!input)
                    return null;
                return Parser.interfaceToJson(Parser.rawToInterface(input, driver), {
                    driver,
                    server,
                });
            }
            catch (e) {
                console.log('[ERROR]: failed to deserialize post object', e, 'input:', input, driver, server);
                return null;
            }
        }
    }
    static parseApiResponse(input, driver, server) {
        if (Array.isArray(input)) {
            return Ok({
                items: Parser.parse(input, driver, server),
                maxId: null,
                minId: null,
                isLoaded: true,
            });
        }
        console.log('[WARN]: failed to identify shape of the input');
        return Err(ApiErrorCode.PARSING_FAILED);
    }
}
class Inspector {
    static getContentTarget(input) {
        if (!input) {
            console.log('[WARN]: trying to obtain target post for', input);
            return input;
        }
        if (input.meta.isBoost && !input.boostedFrom) {
            console.log('[WARN]: original object not available for a repost', input);
            return input;
        }
        return input.meta.isBoost
            ? input.content.raw || input.content.media.length > 0
                ? input
                : input.boostedFrom
            : input;
    }
    static isQuoteObject(input) {
        return (input?.meta?.isBoost &&
            (input?.content?.raw || input?.content?.media?.length > 0));
    }
    static isLiked(input) {
        if (!input)
            return false;
        return !!input.atProto?.viewer?.like || input.interaction.liked;
    }
    static isShared(input) {
        if (!input)
            return false;
        return !!input.atProto?.viewer?.repost || input.interaction.boosted;
    }
}
class Resolver {
    static mentionItemsToWebfinger(handle, items) {
        const parts = handle.split('@').filter(Boolean);
        if (parts.length === 1) {
            const match = items.find((o) => o.acct?.startsWith(parts[0]) && o.url?.endsWith(parts[0]));
            if (match) {
                return {
                    use: 'userId',
                    userId: match.id,
                };
            }
        }
        const match = items.find((o) => o.acct === `${parts[0]}@${parts[1]}`);
        if (match) {
            return {
                use: 'userId',
                userId: match.id,
            };
        }
        return null;
    }
}
export { Parser as PostParser, Inspector as PostInspector, Resolver as PostResolver, postObjectSchema, };
//# sourceMappingURL=post.js.map