import { z } from 'zod';
import { postObjectSchema } from './post.js';
import { appUserObjectSchema, UserParser } from './user.js';
import { PostParser } from './post.js';
import { DriverNotificationType, KNOWN_SOFTWARE } from '../data/driver.js';
const appNotificationGroupedUserItemSchema = z.object({
    item: appUserObjectSchema,
    types: z.array(z.string()),
    extraData: z.any(),
});
export const appNotificationObjectSchema = z.object({
    id: z.string(),
    type: z.string(),
    createdAt: z.coerce.date(),
    user: appUserObjectSchema.nullable(),
    post: postObjectSchema.nullable(),
    extraData: z.any(),
    read: z.boolean(),
    users: z.array(appNotificationGroupedUserItemSchema).optional(),
});
class Parser {
    static rawToObject(input, driver, server) {
        if (driver === KNOWN_SOFTWARE.MASTODON) {
            return {
                id: input.id,
                type: input['visibility'] === 'direct'
                    ? DriverNotificationType.CHAT
                    : DriverNotificationType.MENTION,
                createdAt: input.createdAt,
                user: UserParser.parse(input.account, driver, server),
                post: PostParser.parse(input, driver, server) || null,
                extraData: {},
                read: false,
            };
        }
        const obj = {
            id: input.id,
            type: input.type,
            createdAt: input.createdAt,
            groupKey: input.groupKey,
            user: UserParser.parse(input.account || input.user, driver, server),
            post: PostParser.parse(input.status || input.data || input.note, driver, server),
            extraData: input.reaction,
            read: input.isRead || false,
        };
        const { data, error, success } = appNotificationObjectSchema.safeParse(obj);
        if (!success) {
            console.log('[ERROR]: notification item dto validation failed', error);
            console.log(input);
            return null;
        }
        return data;
    }
    static parse(input, driver, server) {
        if (Array.isArray(input)) {
            return input
                .map((o) => Parser.rawToObject(o, driver, server))
                .filter((o) => !!o);
        }
        else {
            return Parser.rawToObject(input, driver, server);
        }
    }
}
export { Parser as NotificationParser };
//# sourceMappingURL=notification.js.map