import { appUserObjectSchema, UserParser } from './user.js';
import { z } from 'zod';
import { KNOWN_SOFTWARE } from '../data/driver.js';
const feedObjectSchema = z.object({
    uri: z.string(),
    cid: z.string(),
    did: z.string(),
    creator: appUserObjectSchema,
    displayName: z.string(),
    description: z.string().optional(),
    avatar: z.string().optional(),
    likeCount: z.number().int().optional(),
    labels: z.array(z.any()).optional(),
    viewer: z.object({
        like: z.string().optional(),
    }),
    indexedAt: z.date(),
});
class Parser {
    static rawToObject(input, driver, server) {
        if (driver !== KNOWN_SOFTWARE.BLUESKY)
            return null;
        const _input = input;
        const dto = {
            uri: _input.uri,
            cid: _input.cid,
            did: _input.did,
            creator: UserParser.parse(_input.creator, driver, server),
            displayName: _input.displayName,
            description: _input.description,
            avatar: _input.avatar,
            likeCount: _input.likeCount,
            labels: _input.labels,
            viewer: {
                like: _input.viewer?.like,
            },
            indexedAt: new Date(_input.indexedAt),
        };
        const { data, error, success } = feedObjectSchema.safeParse(dto);
        if (!success) {
            console.log('[ERROR]: feed dto validation failed', error);
            return null;
        }
        return data;
    }
    static parse(input, driver, server) {
        if (Array.isArray(input)) {
            return input
                .map((o) => Parser.rawToObject(o, driver, server))
                .filter((o) => !!o);
        }
        else {
            return Parser.rawToObject(input, driver, server);
        }
    }
}
export { Parser as FeedParser };
//# sourceMappingURL=feed.js.map