import { z } from 'zod';
import { RandomUtil } from '../utils/index.js';
import { KNOWN_SOFTWARE } from '../data/driver.js';
const appMessageObjectSchema = z.object({
    uuid: z.string(),
    id: z.string(),
    sender: z.object({
        id: z.string(),
    }),
    content: z.object({
        raw: z.string().nullable().optional(),
    }),
    createdAt: z.string(),
});
class Parser {
    static export(input, driver, server) {
        if (!input)
            return null;
        if (driver !== KNOWN_SOFTWARE.BLUESKY)
            return null;
        return {
            uuid: RandomUtil.nanoId(),
            id: input.id,
            sender: {
                id: input?.sender?.did,
            },
            createdAt: input.sentAt,
            content: {
                raw: input.text,
            },
        };
    }
    static parse(input, driver, server) {
        if (Array.isArray(input)) {
            return input
                .filter((o) => !!o)
                .map((o) => Parser.rawToJson(o, {
                driver,
                server,
            }));
        }
        else {
            return Parser.rawToJson(input, {
                driver,
                server,
            });
        }
    }
    static rawToJson(input, { driver, server, }) {
        if (!input)
            return null;
        if (driver !== KNOWN_SOFTWARE.BLUESKY)
            return null;
        const exported = Parser.export(input, driver, server);
        const { data, error, success } = appMessageObjectSchema.safeParse(exported);
        if (!success) {
            console.log('[ERROR]: status item dto validation failed', error);
            console.log('[INFO]: generated object', exported);
            input.print();
            return null;
        }
        return data;
    }
}
export { Parser as ChatParser };
//# sourceMappingURL=chat.js.map