import { ApiErrorCode } from '../types/result.types.js';
import { CasingUtil } from '../utils/casing.js';
import { BaseUrlNormalizationService } from '../utils/urls.js';
class FetchWrapper {
    baseUrl;
    token;
    requestHeader;
    constructor(urlLike, token) {
        this.baseUrl = BaseUrlNormalizationService.appendHttps(urlLike);
        this.token = token;
        this.requestHeader = this.token
            ? {
                'Content-Type': 'application/json',
                Authorization: `Bearer ${this.token}`,
            }
            : {
                'Content-Type': 'application/json',
            };
        return this;
    }
    static create(urlLike, token) {
        return new FetchWrapper(urlLike, token);
    }
    static cleanObject(obj) {
        Object.keys(obj).forEach((key) => {
            if (obj[key] === null || obj[key] === undefined) {
                delete obj[key];
            }
        });
        let typesOverride = obj['types[]'];
        const retval = CasingUtil.snakeCaseKeys(obj);
        if (typesOverride) {
            delete retval['types'];
            retval['types[]'] = typesOverride;
        }
        return retval;
    }
    withQuery(endpoint, query) {
        if (!query)
            return `${this.baseUrl}${endpoint}`;
        if (query['types[]'] !== undefined) {
            const sample = FetchWrapper.cleanObject(query);
            const items = sample['types[]'].split(';');
            delete sample['query[]'];
            const params = new URLSearchParams(FetchWrapper.cleanObject(sample));
            for (const item of items) {
                params.append('types[]', item);
            }
            console.log(`${this.baseUrl}${endpoint}?` + params.toString());
            return `${this.baseUrl}${endpoint}?` + params.toString();
        }
        return (`${this.baseUrl}${endpoint}?` +
            new URLSearchParams(FetchWrapper.cleanObject(query)));
    }
    async getCamelCaseWithLinkPagination(endpoint, query) {
        endpoint = this.withQuery(endpoint, query);
        return await fetch(endpoint, {
            method: 'GET',
            headers: this.requestHeader,
        })
            .then(async (response) => {
            if (!response.ok) {
                throw new Error(JSON.stringify({
                    status: response.status,
                    statusText: response.statusText,
                }));
            }
            const { minId, maxId } = DhaagaApiUtils.extractPaginationFromLinkHeader(response.headers);
            const _data = CasingUtil.camelCaseKeys(await response.json());
            return {
                data: {
                    data: _data,
                    minId,
                    maxId,
                },
            };
        })
            .catch((e) => {
            console.log(e);
            return {
                error: {
                    code: ApiErrorCode.UNKNOWN_ERROR,
                    message: e,
                },
            };
        });
    }
    async getCamelCase(endpoint, query) {
        endpoint = this.withQuery(endpoint, query);
        return await fetch(endpoint, {
            method: 'GET',
            headers: this.requestHeader,
        })
            .then(async (response) => {
            if (!response.ok) {
                throw new Error(JSON.stringify({
                    status: response.status,
                    statusText: response.statusText,
                }));
            }
            DhaagaApiUtils.extractPaginationFromLinkHeader(response.headers);
            const data = CasingUtil.camelCaseKeys(await response.json());
            return { data };
        })
            .catch((e) => {
            return {
                error: {
                    code: ApiErrorCode.UNKNOWN_ERROR,
                    message: e,
                },
            };
        });
    }
    async post(endpoint, body, opts) {
        endpoint = `${this.baseUrl}${endpoint}`;
        return await fetch(endpoint, {
            method: 'POST',
            headers: this.token
                ? {
                    'Content-Type': 'application/json',
                    Authorization: `Bearer ${this.token}`,
                }
                : {
                    'Content-Type': 'application/json',
                },
            body: JSON.stringify(body),
        })
            .then(async (response) => {
            if (!response.ok) {
                throw new Error(JSON.stringify({
                    status: response.status,
                    statusText: response.statusText,
                }));
            }
            return { data: await response.json() };
        })
            .catch((e) => {
            return {
                error: {
                    code: ApiErrorCode.UNKNOWN_ERROR,
                    message: e,
                },
            };
        });
    }
    static applyQueriesToRequestUrl(url, opts) {
        return opts.queries
            ? `${opts.baseURL}${url}?` +
                new URLSearchParams(FetchWrapper.cleanObject(opts.queries))
            : `${opts.baseURL}${url}?`;
    }
    static buildRequestInitObject(opts) {
        if (!opts)
            return {};
        return {
            ...opts,
            headers: opts.accessToken
                ? {
                    'Content-Type': 'application/json',
                    Authorization: `Bearer ${opts.accessToken}`,
                }
                : {
                    'Content-Type': 'application/json',
                },
        };
    }
    static execute() { }
    async get(endpoint, opts) {
        endpoint = FetchWrapper.applyQueriesToRequestUrl(endpoint, {
            ...opts,
            baseURL: opts?.baseURL || this.baseUrl,
        });
        try {
            const response = await fetch(endpoint, FetchWrapper.buildRequestInitObject({ ...opts, method: 'GET' }));
            if (!response.ok) {
                let errorBody;
                let message = response.statusText;
                let errorCode = ApiErrorCode.UNKNOWN_ERROR;
                try {
                    errorBody = await response.json();
                    if (typeof errorBody?.error === 'string')
                        message = errorBody.error;
                    if (typeof errorBody?.error?.message === 'string')
                        message = errorBody.error.message;
                }
                catch {
                    errorBody = await response.text();
                    console.log('via body', errorBody);
                }
                switch (response.status) {
                    case 401: {
                        errorCode = ApiErrorCode.UNAUTHORIZED;
                        break;
                    }
                }
                return {
                    error: {
                        code: errorCode,
                        message,
                    },
                };
            }
            let data = await response.json();
            if (opts?.transformResponse === 'snake') {
                data = CasingUtil.snakeCaseKeys(data);
            }
            else if (opts?.transformResponse === 'camel') {
                data = CasingUtil.camelCaseKeys(data);
            }
            return data;
        }
        catch (e) {
            return {
                error: {
                    code: ApiErrorCode.UNKNOWN_ERROR,
                    message: e,
                },
            };
        }
    }
}
class DhaagaApiUtils {
    static extractPaginationFromLinkHeader(headers) {
        const linkHeader = headers?.map?.link;
        const maxIdRegex = /max_id=([0-9]+)/;
        const minIdRegex = /min_id=([0-9]+)/;
        let maxId = null;
        let minId = null;
        if (minIdRegex.test(linkHeader)) {
            const minMatch = linkHeader.match(minIdRegex);
            minId = minMatch[1];
        }
        if (maxIdRegex.test(linkHeader)) {
            const maxMatch = linkHeader.match(maxIdRegex);
            maxId = maxMatch[1];
        }
        return { minId, maxId };
    }
}
export default FetchWrapper;
//# sourceMappingURL=custom-fetch.js.map