import { errorBuilder } from '../_router/dto/api-responses.dto.js';
import { ApiErrorCode } from '../../../types/result.types.js';
import { MegalodonPleromaWrapper } from '../../../custom-clients/custom-clients.js';
import { CasingUtil } from '../../../utils/casing.js';
import { Err, Ok } from '../../../utils/index.js';
export class PleromaStatusesRouter {
    direct;
    client;
    constructor(forwarded) {
        this.direct = forwarded;
        this.client = MegalodonPleromaWrapper.create(forwarded.baseUrl, forwarded.token);
    }
    async get(id) {
        const response = await this.client.client.getStatus(id);
        if (response.status !== 200) {
            console.log('[ERROR]: failed to get status', response.statusText);
        }
        return {
            data: CasingUtil.camelCaseKeys(response.data),
        };
    }
    async create(dto) {
        const response = await this.client.client.postStatus(dto.status, {
            language: dto.language,
            visibility: dto.mastoVisibility,
            in_reply_to_id: dto.inReplyToId,
            sensitive: dto.sensitive,
            spoiler_text: dto.spoilerText,
            media_ids: dto.mediaIds || [],
        });
        if (response.status !== 200) {
            console.log('[ERROR]: failed to create status', response.statusText);
        }
        return { data: CasingUtil.camelCaseKeys(response.data) };
    }
    async delete(id) {
        const data = await this.client.client.deleteStatus(id);
        if (data.status === 200 || data.status === 204) {
            return { success: true, deleted: true };
        }
        return { success: false, deleted: false };
    }
    async getReactions(id) {
        const data = await this.client.client.getEmojiReactions(id);
        return { data: CasingUtil.camelCaseKeys(data.data) };
    }
    async getReactionDetails(postId, reactionId) {
        const data = await this.client.client.getEmojiReactions(postId);
        if (data.status !== 200) {
            console.log('[ERROR]: failed to get reaction details', data.statusText);
            return errorBuilder(data.statusText);
        }
        return { data: CasingUtil.camelCaseKeys(data.data) };
    }
    async addReaction(id, shortCode) {
        const data = await this.client.client.createEmojiReaction(id, shortCode);
        if (data.status !== 200) {
            console.log('[ERROR]: failed to add reaction', data.statusText);
            return errorBuilder(data.statusText);
        }
        return { data: CasingUtil.camelCaseKeys(data.data) };
    }
    async removeReaction(id, shortCode) {
        const data = await this.client.client.deleteEmojiReaction(id, shortCode);
        if (data.status !== 200) {
            console.log('[ERROR]: failed to remove reaction', data.statusText);
            return errorBuilder(data.statusText);
        }
        return { data: CasingUtil.camelCaseKeys(data.data) };
    }
    async bookmark(id) {
        const data = await this.client.client.bookmarkStatus(id);
        return Ok({ state: data.data.bookmarked });
    }
    async unBookmark(id) {
        const data = await this.client.client.unbookmarkStatus(id);
        return Ok({ state: data.data.bookmarked });
    }
    async like(id) {
        try {
            const data = await this.client.client.favouriteStatus(id);
            return Ok({
                state: !!data.data.favourited,
                counter: data.data.favourites_count,
            });
        }
        catch (e) {
            return Err(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async removeLike(id) {
        try {
            const data = await this.client.client.unfavouriteStatus(id);
            return Ok({
                state: !!data.data.favourited,
                counter: data.data.favourites_count,
            });
        }
        catch (e) {
            return Err(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async getContext(id) {
        const data = await this.client.client.getStatusContext(id);
        return { data: CasingUtil.camelCaseKeys(data.data) };
    }
    async boost(id) {
        const data = await this.client.client.reblogStatus(id);
        return { data: data.data };
    }
    async removeBoost(id) {
        const data = await this.client.client.unreblogStatus(id);
        return { data: data.data };
    }
}
//# sourceMappingURL=statuses.js.map