import { errorBuilder, notImplementedErrorBuilder, } from '../_router/dto/api-responses.dto.js';
import { BaseAccountsRouter } from '../default/accounts.js';
import { ApiErrorCode } from '../../../types/result.types.js';
import { MegalodonPleromaWrapper } from '../../../custom-clients/custom-clients.js';
import { CasingUtil } from '../../../utils/casing.js';
import { Err, Ok } from '../../../utils/index.js';
export class PleromaAccountsRouter extends BaseAccountsRouter {
    direct;
    client;
    constructor(forwarded) {
        super();
        this.direct = forwarded;
        this.client = MegalodonPleromaWrapper.create(forwarded.baseUrl, forwarded.token);
    }
    async get(id) {
        const data = await this.client.client.getAccount(id);
        if (data.status !== 200)
            return errorBuilder(data.statusText);
        return { data: CasingUtil.camelCaseKeys(data.data) };
    }
    async lookup(webfinger) {
        const data = await this.client.client.lookupAccount(webfinger.host
            ? `${webfinger.username}@${webfinger.host}`
            : webfinger.username);
        if (data.status !== 200)
            return Err(data.statusText);
        return Ok(data.data);
    }
    async statuses(id, query) {
        try {
            const data = await this.client.client.getAccountStatuses(id, CasingUtil.snakeCaseKeys(query));
            return Ok(CasingUtil.camelCaseKeys(data.data));
        }
        catch (e) {
            console.log('[ERROR]: getting pleroma user timeline', e);
            return Err(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async relationships(ids) {
        const data = await this.client.client.getRelationships(ids);
        if (data.status !== 200) {
            return errorBuilder(data.statusText);
        }
        return { data: CasingUtil.camelCaseKeys(data.data) };
    }
    async likes(query) {
        const { data: _data, error } = await this.direct.getCamelCaseWithLinkPagination('/api/v1/favourites', query);
        if (!_data || error) {
            return notImplementedErrorBuilder();
        }
        return {
            data: _data,
        };
    }
    async bookmarks(query) {
        const { data: _data, error } = await this.direct.getCamelCaseWithLinkPagination('/api/v1/bookmarks', query);
        if (!_data || error) {
            return notImplementedErrorBuilder();
        }
        return {
            data: _data,
        };
    }
    async follow(id) {
        const data = await this.client.client.followAccount(id);
        if (data.status !== 200) {
            return errorBuilder(data.statusText);
        }
        return { data: CasingUtil.camelCaseKeys(data.data) };
    }
    async unfollow(id) {
        const data = await this.client.client.unfollowAccount(id);
        if (data.status !== 200) {
            return errorBuilder(data.statusText);
        }
        return { data: CasingUtil.camelCaseKeys(data.data) };
    }
    async followers(query) {
        try {
            const { id, ...rest } = query;
            const { data: _data, error } = await this.direct.getCamelCaseWithLinkPagination(`/api/v1/accounts/${id}/followers`, rest);
            if (error) {
                return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
            }
            return { data: _data };
        }
        catch (e) {
            console.log(e);
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async followings(query) {
        try {
            const { id, ...rest } = query;
            const { data: _data, error } = await this.direct.getCamelCaseWithLinkPagination(`/api/v1/accounts/${id}/following`, rest);
            if (error) {
                return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
            }
            return { data: _data };
        }
        catch (e) {
            console.log(e);
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
}
//# sourceMappingURL=accounts.js.map