import { errorBuilder } from '../_router/dto/api-responses.dto.js';
import { ApiErrorCode } from '../../../types/result.types.js';
import { MisskeyJsWrapper } from '../../../custom-clients/custom-clients.js';
import { Err, Ok } from '../../../utils/index.js';
export class MisskeyStatusesRouter {
    direct;
    client;
    constructor(forwarded) {
        this.direct = forwarded;
        this.client = MisskeyJsWrapper.create(forwarded.baseUrl, forwarded.token);
    }
    async create(dto) {
        try {
            console.log('dto', {
                lang: dto.language,
                visibility: dto.misskeyVisibility,
                replyId: dto.inReplyToId,
                text: dto.status,
                visibleUserIds: dto.misskeyVisibility === 'specified'
                    ? dto.visibleUserIds || []
                    : undefined,
                fileIds: dto.mediaIds || [],
                cw: dto.spoilerText || null,
                localOnly: false, // reactionAcceptance: null,
                poll: dto.poll || null,
                scheduledAt: null,
            });
            const data = await this.client.client.request('notes/create', {
                lang: dto.language,
                visibility: dto.misskeyVisibility,
                replyId: dto.inReplyToId,
                text: dto.status,
                visibleUserIds: dto.misskeyVisibility === 'specified'
                    ? dto.visibleUserIds || []
                    : undefined,
                fileIds: dto.mediaIds.length > 0 ? dto.mediaIds || [] : undefined,
                cw: dto.spoilerText || undefined,
                localOnly: false, // reactionAcceptance: null,
                poll: dto.poll || null,
                scheduledAt: null,
            });
            return { data: data };
        }
        catch (e) {
            if (e.code) {
                return errorBuilder(e);
            }
            console.log(e);
            return errorBuilder(ApiErrorCode.UNAUTHORIZED);
        }
    }
    async delete(id) {
        try {
            await this.client.client.request('notes/delete', {
                noteId: id,
            });
            return { success: true, deleted: true };
        }
        catch (e) {
            if (e.code) {
                return { success: false, deleted: false };
            }
            return { success: false, deleted: false };
        }
    }
    async get(id) {
        const data = await this.client.client.request('notes/show', { noteId: id });
        return { data };
    }
    async getReactions(postId) {
        const data = await this.client.client.request('notes/reactions', {
            noteId: postId,
        });
        return { data };
    }
    async getReactionDetails(postId, reactionId) {
        try {
            const data = await this.client.client.request('notes/reactions', {
                noteId: postId,
                type: reactionId,
            });
            return { data };
        }
        catch (e) {
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async addReaction(postId, reactionId) {
        try {
            await this.client.client.request('notes/reactions/create', {
                noteId: postId,
                reaction: reactionId,
            });
            return { data: { success: true, reacted: true, id: reactionId } };
        }
        catch (e) {
            if (e.code)
                return errorBuilder(e);
            console.log('[ERROR]: failed to add reaction', reactionId, e);
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async removeReaction(postId, reactionId) {
        try {
            await this.client.client.request('notes/reactions/delete', {
                noteId: postId,
                reaction: reactionId,
            });
            return { data: { success: false, reacted: false, id: reactionId } };
        }
        catch (e) {
            if (e.code)
                return errorBuilder(e);
            console.log('[ERROR]: failed to remove reaction', reactionId, e);
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async getState(id) {
        try {
            const data = await this.client.client.request('notes/state', {
                noteId: id,
            });
            return { data };
        }
        catch (e) {
            if (e.code) {
                return errorBuilder(e);
            }
            console.log(e);
            return errorBuilder(ApiErrorCode.UNAUTHORIZED);
        }
    }
    async bookmark(id) {
        try {
            await this.client.client.request('notes/favorites/create', {
                noteId: id,
            });
            return Ok({ state: true });
        }
        catch (e) {
            if (e.code) {
                if (e.code === 'ALREADY_FAVORITED')
                    return Ok({ state: true });
                return Err(e.code);
            }
            return Err(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async unBookmark(id) {
        try {
            await this.client.client.request('notes/favorites/delete', {
                noteId: id,
            });
            return Ok({ state: false });
        }
        catch (e) {
            if (e.code) {
                if (e.code === 'NOT_FAVORITED')
                    return Ok({ state: false });
                return Err(e.code);
            }
            return Err(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async renotes(id) {
        try {
            const data = await this.client.client.request('notes/renotes', {
                noteId: id,
            });
            return { data };
        }
        catch (e) {
            if (e.code) {
                return errorBuilder(e);
            }
            console.log(e);
            return errorBuilder(ApiErrorCode.UNAUTHORIZED);
        }
    }
    async favourite(id) {
        const { error } = await this.direct.post('/api/notes/like', { noteId: id }, {});
        if (error)
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        return { data: { success: true, isFavourited: true } };
    }
    async like(id) {
        const { error } = await this.direct.post('/api/notes/like', { noteId: id }, {});
        if (error)
            return Err(ApiErrorCode.REMOTE_SERVER_ERROR);
        return Ok({ state: true });
    }
    async removeLike(id) {
        return Err(ApiErrorCode.OPERATION_UNSUPPORTED);
    }
    async renote(dto) {
        try {
            const data = await this.client.client.request('notes/create', dto);
            return {
                data: {
                    success: true,
                    renoted: true,
                    post: data,
                },
            };
        }
        catch (e) {
            if (e.code) {
                return errorBuilder(e);
            }
            console.log(e);
            return errorBuilder(ApiErrorCode.UNAUTHORIZED);
        }
    }
    async unrenote(id) {
        try {
            await this.client.client.request('notes/unrenote', {
                noteId: id,
            });
            return { data: { success: true, renoted: false } };
        }
        catch (e) {
            if (e.code) {
                return errorBuilder(e);
            }
            console.log(e);
            return errorBuilder(ApiErrorCode.UNAUTHORIZED);
        }
    }
    async getContext(id, limit) {
        try {
            const parents = this.client.client.request('notes/conversation', {
                noteId: id,
                limit: limit || 40,
            });
            const children = this.client.client.request('notes/children', {
                noteId: id,
                showQuotes: false,
                limit: limit || 40,
            });
            return new Promise((resolve, reject) => {
                Promise.all([parents, children])
                    .then(([_parents, _children]) => {
                    resolve({
                        data: {
                            ancestors: _parents,
                            descendants: _children,
                        },
                    });
                })
                    .catch((e) => {
                    return resolve({ data: { ancestors: [], descendants: [] } });
                });
            });
        }
        catch (e) {
            console.log(e);
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
}
//# sourceMappingURL=statuses.js.map