import { errorBuilder, notImplementedErrorBuilder, } from '../_router/dto/api-responses.dto.js';
import { BaseAccountsRouter } from '../default/accounts.js';
import { ApiErrorCode } from '../../../types/result.types.js';
import { MisskeyJsWrapper } from '../../../custom-clients/custom-clients.js';
import { Ok } from '../../../utils/index.js';
export class MisskeyAccountsRouter extends BaseAccountsRouter {
    direct;
    client;
    constructor(forwarded) {
        super();
        this.direct = forwarded;
        this.client = MisskeyJsWrapper.create(forwarded.baseUrl, forwarded.token);
    }
    async statuses(id, query) {
        const data = await this.client.client.request('users/notes', {
            ...query,
            withFiles: !!query.onlyMedia ? query.onlyMedia : undefined,
        });
        return Ok(data);
    }
    async relationships(ids) {
        return notImplementedErrorBuilder();
    }
    async get(id) {
        try {
            const data = await this.client.client.request('users/show', {
                userId: id,
            });
            return { data };
        }
        catch (e) {
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async getMany(ids) {
        try {
            const data = await this.client.client.request('users/show', {
                userIds: ids,
            });
            return { data };
        }
        catch (e) {
            if (e.code) {
                return errorBuilder(e.code);
            }
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async follow(id, opts) {
        try {
            const data = await this.client.client.request('following/create', {
                userId: id,
                ...opts,
            });
            return { data };
        }
        catch (e) {
            if (e.code) {
                return errorBuilder(e.code);
            }
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async unfollow(id) {
        try {
            const data = await this.client.client.request('following/delete', {
                userId: id,
            });
            return { data };
        }
        catch (e) {
            if (e.code) {
                return errorBuilder(e.code);
            }
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async block(id) {
        try {
            const data = await this.client.client.request('blocking/create', {
                userId: id,
            });
            return { data };
        }
        catch (e) {
            console.log(e);
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async unblock(id) {
        try {
            const data = await this.client.client.request('blocking/delete', {
                userId: id,
            });
            return { data };
        }
        catch (e) {
            console.log(e);
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async findByWebfinger({ username, host, }) {
        try {
            const data = await this.client.client.request('users/show', {
                username,
                host,
            });
            return { data };
        }
        catch (e) {
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async findByUserId(id) {
        try {
            const data = await this.client.client.request('users/show', {
                userId: id,
            });
            return { data };
        }
        catch (e) {
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async findByUserIds(ids) {
        if (ids.length === 0)
            return { data: [] };
        try {
            const data = await this.client.client.request('users/show', {
                userIds: ids,
            });
            return { data };
        }
        catch (e) {
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async renoteMute(id) {
        try {
            await this.client.client.request('renote-mute/create', {
                userId: id,
            });
            return { data: { renoteMuted: true } };
        }
        catch (e) {
            console.log(e);
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async renoteUnmute(id) {
        try {
            await this.client.client.request('renote-mute/delete', {
                userId: id,
            });
            return { data: { renoteMuted: false } };
        }
        catch (e) {
            console.log(e);
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async likes(query) {
        return errorBuilder(ApiErrorCode.FEATURE_UNSUPPORTED);
    }
    async bookmarks(query) {
        try {
            const { data, error } = await this.direct.post('/api/i/favorites', {
                limit: query.limit,
                untilId: !!query.maxId ? query.maxId : undefined,
            }, {});
            if (error) {
                return {
                    data: {
                        data: [],
                    },
                };
            }
            let maxId = null;
            let minId = null;
            if (data.length > 0) {
                maxId = data[data.length - 1].id;
                minId = data[0].id;
            }
            return {
                data: {
                    data: data.map((o) => o.note),
                    maxId,
                    minId,
                },
            };
        }
        catch (e) {
            if (e.code) {
                return errorBuilder(ApiErrorCode.UNAUTHORIZED);
            }
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async followers(query) {
        try {
            const data = await this.client.client.request('users/followers', {
                allowPartial: true,
                limit: query.limit,
                userId: query.id,
                untilId: !!query.maxId ? query.maxId : undefined,
            });
            return { data: { data } };
        }
        catch (e) {
            if (e.code) {
                return errorBuilder(e.code);
            }
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async followings(query) {
        try {
            const data = await this.client.client.request('users/following', {
                allowPartial: true,
                limit: query.limit,
                userId: query.id,
                untilId: !!query.maxId ? query.maxId : undefined,
            });
            return { data: { data } };
        }
        catch (e) {
            if (e.code) {
                return errorBuilder(e.code);
            }
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
}
//# sourceMappingURL=accounts.js.map