import { errorBuilder } from '../_router/dto/api-responses.dto.js';
import { ApiErrorCode } from '../../../types/result.types.js';
import { Err, Ok } from '../../../utils/index.js';
import { getBskyAgent } from '../../../utils/atproto.js';
class BlueskyStatusesRouter {
    dto;
    constructor(dto) {
        this.dto = dto;
    }
    bookmark(id) {
        return Promise.resolve(undefined);
    }
    async create(dto) {
        const agent = getBskyAgent(this.dto);
        try {
            const data = await agent.post({
                text: dto.status,
            });
            return { data };
        }
        catch (e) {
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async delete(id) {
        const agent = getBskyAgent(this.dto);
        try {
            await agent.deletePost(id);
            return { success: true, deleted: true };
        }
        catch (e) {
            return { success: false, deleted: false };
        }
    }
    async get(id) {
        const agent = getBskyAgent(this.dto);
        try {
            const data = await agent.getPostThread({
                uri: id,
                depth: 1,
            });
            return { data };
        }
        catch (e) {
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async getContext(id, limit) {
        const agent = getBskyAgent(this.dto);
        try {
            const data = await agent.getPostThread({
                uri: id,
                depth: limit || 10,
            });
            return { data };
        }
        catch (e) {
            console.log('[ERROR]: bluesky', e);
            return errorBuilder();
        }
    }
    async like(uri, cid) {
        if (!cid)
            return Err(ApiErrorCode.INVALID_INPUT);
        const agent = getBskyAgent(this.dto);
        try {
            const result = await agent.like(uri, cid);
            return Ok({ state: true, uri: result.uri });
        }
        catch (e) {
            return Err(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async removeLike(uri, cid) {
        if (!cid)
            return Err(ApiErrorCode.INVALID_INPUT);
        const agent = getBskyAgent(this.dto);
        try {
            await agent.deleteLike(uri);
            return Ok({ state: false });
        }
        catch (e) {
            return Err(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    unBookmark(id) {
        return Promise.resolve(undefined);
    }
    async getConvoForMembers(members) {
        const agent = getBskyAgent(this.dto);
        try {
            const data = await agent.chat.bsky.convo.getConvoForMembers({
                members,
            });
            return { data };
        }
        catch (e) {
            console.log('[ERROR]: bluesky', e);
            return errorBuilder();
        }
    }
    async getConvo(id) {
        const agent = getBskyAgent(this.dto);
        try {
            const data = await agent.chat.bsky.convo.getConvo({
                convoId: id,
            });
            return { data };
        }
        catch (e) {
            console.log('[ERROR]: bluesky', e);
            return errorBuilder();
        }
    }
    async sendMessage(id, msg) {
        const agent = getBskyAgent(this.dto);
        try {
            const data = await agent.chat.bsky.convo.sendMessage({
                convoId: id,
                message: { text: msg },
            });
            return { data };
        }
        catch (e) {
            console.log('[ERROR]: bluesky', e);
            return errorBuilder();
        }
    }
    async atProtoLike(uri, cid) {
        const agent = getBskyAgent(this.dto);
        try {
            const result = await agent.like(uri, cid);
            return Ok({ state: true, uri: result.uri });
        }
        catch (e) {
            return Err(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async atProtoDeleteLike(uri) {
        const agent = getBskyAgent(this.dto);
        try {
            await agent.deleteLike(uri);
            return Ok({ state: false });
        }
        catch (e) {
            return Err(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async atProtoRepost(uri, cid) {
        try {
            const agent = getBskyAgent(this.dto);
            const result = await agent.repost(uri, cid);
            return { success: true, liked: true, uri: result.uri };
        }
        catch (e) {
            console.log(e);
            return { success: false };
        }
    }
    async atProtoDeleteRepost(uri) {
        const agent = getBskyAgent(this.dto);
        try {
            await agent.deleteRepost(uri);
            return { success: true, liked: false };
        }
        catch (e) {
            console.log(e);
            return { success: false };
        }
    }
}
export default BlueskyStatusesRouter;
//# sourceMappingURL=statuses.js.map