import { UserParser } from '../../../../parsers/user.js';
import { Err, Ok } from '../../../../utils/index.js';
import { ApiErrorCode } from '../../../../types/result.types.js';
import { DriverService } from '../../../../services/driver.js';
import { KeyExtractorUtil } from '../../../../utils/key-extractor.js';
class Route {
    client;
    constructor(client) {
        this.client = client;
    }
    async findOne(query) {
        const driver = this.client.driver;
        const server = this.client.server;
        switch (query.use) {
            case 'did': {
                const { data, error } = await this.client.accounts.get(query.did);
                if (error)
                    return Err(ApiErrorCode.UNKNOWN_ERROR);
                return Ok(UserParser.parse(data.data, driver, server));
            }
            case 'userId': {
                const { data, error } = await this.client.accounts.get(query.userId);
                if (error)
                    return Err(ApiErrorCode.UNKNOWN_ERROR);
                return Ok(UserParser.parse(data.data ? data.data : data, driver, server));
            }
            case 'handle': {
                if (DriverService.supportsAtProto(driver)) {
                    const { data: didData, error: didError } = await this.client.accounts.getDid(query.handle);
                    if (didError)
                        return Err('E_Failed_Did_Lookup');
                    const { data, error } = await this.client.accounts.get(didData?.data?.did);
                    if (error)
                        throw new Error('Failed to fetch user for AtProto');
                    return Ok(UserParser.parse(data.data, driver, server));
                }
                else if (DriverService.supportsMastoApiV1(this.client.driver)) {
                    const res = await this.client.accounts.lookup({
                        username: query.handle,
                        host: null,
                    });
                    if (res.isErr())
                        return Err(ApiErrorCode.UNKNOWN_ERROR);
                    return Ok(UserParser.parse(res.unwrap(), driver, server));
                }
                else if (DriverService.supportsMisskeyApi(this.client.driver)) {
                    return Err(ApiErrorCode.OPERATION_UNSUPPORTED);
                }
                else {
                    return Err(ApiErrorCode.OPERATION_UNSUPPORTED);
                }
            }
            case 'webfinger': {
                if (DriverService.supportsMastoApiV1(this.client.driver)) {
                    const res = await this.client.accounts.lookup(query.webfinger);
                    return res.map((o) => UserParser.parse(o, driver, server));
                }
                else if (DriverService.supportsMisskeyApi(this.client.driver)) {
                    const findResult = await this.client.accounts.findByWebfinger(query.webfinger);
                    return Ok(UserParser.parse(findResult.data, driver, server));
                }
                else {
                    return Err(ApiErrorCode.OPERATION_UNSUPPORTED);
                }
            }
            default: {
                return Err(ApiErrorCode.OPERATION_UNSUPPORTED);
            }
        }
    }
    async getFollowers(query) {
        const result = await this.client.accounts.followers(query);
        return Ok(KeyExtractorUtil.getPage(result, (o) => UserParser.parse(o, this.client.driver, this.client.server)));
    }
    async getFollows(query) {
        const result = await this.client.accounts.followings(query);
        return Ok(KeyExtractorUtil.getPage(result, (o) => UserParser.parse(o, this.client.driver, this.client.server)));
    }
}
export { Route as UserRoute };
//# sourceMappingURL=user.js.map