import { PostInspector } from '../../../../parsers/post.js';
import { DriverService } from '../../../../services/driver.js';
import { Err, Ok } from '../../../../utils/index.js';
import { ApiErrorCode } from '../../../../types/result.types.js';
import ActivityPubService from '../../../../services/activitypub.service.js';
import { AtprotoPostService } from '../../../../services/atproto.service.js';
import ActivityPubReactionsService from '../../../../services/activitypub-reactions.service.js';
class Mutator {
    client;
    constructor(client) {
        this.client = client;
    }
    _applyReactionData(input, _data) {
        const target = PostInspector.getContentTarget(input);
        const draft = JSON.parse(JSON.stringify(input));
        if (draft.id === target.id)
            draft.stats.reactions = _data;
        if (draft.boostedFrom && draft.boostedFrom.id === target.id)
            draft.boostedFrom.stats.reactions = _data;
        return draft;
    }
    async addReaction(input, reactionCode) {
        const target = PostInspector.getContentTarget(input);
        try {
            const nextState = await ActivityPubReactionsService.addReaction(this.client, target.id, reactionCode, this.client.driver, () => { });
            return Ok(this._applyReactionData(input, nextState));
        }
        catch (e) {
            return Err(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async removeReaction(input, reactionCode) {
        const target = PostInspector.getContentTarget(input);
        try {
            const nextState = await ActivityPubReactionsService.removeReaction(this.client, target.id, reactionCode, this.client.driver, (ok) => { });
            return Ok(this._applyReactionData(input, nextState));
        }
        catch (e) {
            return Err(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async toggleLike(input) {
        const target = PostInspector.getContentTarget(input);
        let nextState;
        try {
            if (DriverService.supportsAtProto(this.client.driver)) {
                const _api = this.client;
                nextState = target.atProto?.viewer?.like
                    ? await _api.statuses.atProtoDeleteLike(target.atProto?.viewer?.like)
                    : await _api.statuses.atProtoLike(target.meta.uri, target.meta.cid);
            }
            else if (!DriverService.supportsMisskeyApi(this.client.driver)) {
                nextState = target.interaction.liked
                    ? await this.client.statuses.removeLike(target.id)
                    : await this.client.statuses.like(target.id);
            }
            if (nextState === undefined)
                return Err(ApiErrorCode.OPERATION_UNSUPPORTED);
            if (nextState.isErr()) {
                console.log('[WARN]: failed to toggle like', nextState.error);
                return Err(nextState.error);
            }
            const _state = nextState.unwrap();
            const draft = JSON.parse(JSON.stringify(input));
            if (draft.id === target.id) {
                draft.interaction.liked = _state.state;
                draft.stats.likeCount += _state.state ? 1 : -1;
                if (draft.atProto && draft.atProto.viewer)
                    draft.atProto.viewer.like = _state.uri;
            }
            else if (draft.boostedFrom && draft.boostedFrom.id === target.id) {
                draft.boostedFrom.interaction.liked = _state.state;
                draft.boostedFrom.stats.likeCount += _state.state ? 1 : -1;
                if (draft.boostedFrom.atProto && draft.boostedFrom.atProto.viewer)
                    draft.boostedFrom.atProto.viewer.like = _state.uri;
            }
            return Ok(draft);
        }
        catch (e) {
            console.log('[WARN]: failed to toggle like', e);
            return Err(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async toggleShare(input) {
        const target = PostInspector.getContentTarget(input);
        if (DriverService.supportsAtProto(this.client.driver)) {
            if (!target.atProto)
                return input;
            const result = await AtprotoPostService.toggleRepost(this.client, target.meta.uri, target.meta.cid, target.atProto.viewer?.repost);
            if (!result.success)
                return input;
            const draft = JSON.parse(JSON.stringify(input));
            if (draft.id === target.id) {
                draft.interaction.boosted = result.state;
                draft.stats.boostCount += result.state ? 1 : -1;
                if (draft.atProto && draft.atProto.viewer)
                    draft.atProto.viewer.repost = result.uri;
            }
            else if (draft.boostedFrom && draft.boostedFrom?.id === target.id) {
                draft.boostedFrom.interaction.boosted = result.state;
                draft.boostedFrom.stats.boostCount += result.state ? 1 : -1;
                if (draft.boostedFrom.atProto && draft.boostedFrom.atProto.viewer)
                    draft.boostedFrom.atProto.viewer.repost = result.uri;
            }
            return draft;
        }
        else {
            const res = await ActivityPubService.toggleBoost(this.client, target.id, target.interaction.boosted, this.client.driver);
            const draft = JSON.parse(JSON.stringify(input));
            if (draft.id === target.id) {
                draft.interaction.boosted = res !== -1;
                draft.stats.boostCount += res === null ? 0 : res;
            }
            else if (draft.boostedFrom && draft.boostedFrom?.id === target.id) {
                draft.boostedFrom.interaction.boosted = res != -1;
                draft.boostedFrom.stats.boostCount += res === null ? 0 : res;
            }
            return draft;
        }
    }
    async loadBookmarkState(input) {
        const target = PostInspector.getContentTarget(input);
        if (target.state.isBookmarkStateFinal)
            return input;
        try {
            const res = await ActivityPubService.getBookmarkState(this.client, target.id);
            if (res === null)
                return input;
            const draft = JSON.parse(JSON.stringify(input));
            if (draft.id === target.id) {
                draft.interaction.bookmarked = res;
                draft.state.isBookmarkStateFinal = true;
            }
            else if (draft.boostedFrom && draft.boostedFrom.id === target.id) {
                draft.boostedFrom.interaction.bookmarked = res;
                draft.boostedFrom.state.isBookmarkStateFinal = true;
            }
            return draft;
        }
        catch (e) {
            console.log('[WARN]: failed to finalize bookmark state', e);
            return input;
        }
    }
    async toggleBookmark(input) {
        if (!DriverService.canBookmark(this.client.driver))
            return Err(ApiErrorCode.OPERATION_UNSUPPORTED);
        const target = PostInspector.getContentTarget(input);
        try {
            const result = target.interaction.bookmarked
                ? await this.client.statuses.unBookmark(target.id)
                : await this.client.statuses.bookmark(target.id);
            if (result.isErr())
                return Err(ApiErrorCode.UNKNOWN_ERROR);
            const _state = result.unwrap();
            const draft = JSON.parse(JSON.stringify(input));
            if (draft.id === target.id) {
                draft.interaction.bookmarked = _state.state;
                draft.state.isBookmarkStateFinal = true;
            }
            else if (draft.boostedFrom && draft.boostedFrom.id === target.id) {
                draft.boostedFrom.interaction.bookmarked = _state.state;
                draft.boostedFrom.state.isBookmarkStateFinal = true;
            }
            return Ok(draft);
        }
        catch (e) {
            return Err(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
}
export { Mutator as PostMutatorRoute };
//# sourceMappingURL=post.js.map