import axios from 'axios';
import objectAssignDeep from 'object-assign-deep';
import { RequestCanceledError } from '../cancel.js';
import { NO_REDIRECT, DEFAULT_SCOPE, DEFAULT_UA } from '../default.js';
import WebSocket from './web_socket.js';
import NotificationType, { UnknownNotificationTypeError } from '../notification.js';
import PleromaNotificationType from './notification.js';
var PleromaAPI;
(function (PleromaAPI) {
    let Converter;
    (function (Converter) {
        Converter.decodeNotificationType = (t) => {
            switch (t) {
                case PleromaNotificationType.Mention:
                    return NotificationType.Mention;
                case PleromaNotificationType.Reblog:
                    return NotificationType.Reblog;
                case PleromaNotificationType.Favourite:
                    return NotificationType.Favourite;
                case PleromaNotificationType.Follow:
                    return NotificationType.Follow;
                case PleromaNotificationType.Poll:
                    return NotificationType.PollExpired;
                case PleromaNotificationType.PleromaEmojiReaction:
                    return NotificationType.Reaction;
                case PleromaNotificationType.FollowRequest:
                    return NotificationType.FollowRequest;
                case PleromaNotificationType.Update:
                    return NotificationType.Update;
                case PleromaNotificationType.Move:
                    return NotificationType.Move;
                case PleromaNotificationType.Status:
                    return NotificationType.Status;
                default:
                    return new UnknownNotificationTypeError();
            }
        };
        Converter.encodeNotificationType = (t) => {
            switch (t) {
                case NotificationType.Follow:
                    return PleromaNotificationType.Follow;
                case NotificationType.Favourite:
                    return PleromaNotificationType.Favourite;
                case NotificationType.Reblog:
                    return PleromaNotificationType.Reblog;
                case NotificationType.Mention:
                    return PleromaNotificationType.Mention;
                case NotificationType.PollExpired:
                    return PleromaNotificationType.Poll;
                case NotificationType.Reaction:
                    return PleromaNotificationType.PleromaEmojiReaction;
                case NotificationType.FollowRequest:
                    return PleromaNotificationType.FollowRequest;
                case NotificationType.Update:
                    return PleromaNotificationType.Update;
                case NotificationType.Move:
                    return PleromaNotificationType.Move;
                case NotificationType.Status:
                    return PleromaNotificationType.Status;
                default:
                    return new UnknownNotificationTypeError();
            }
        };
        Converter.visibility = (v) => {
            switch (v) {
                case 'public':
                    return 'public';
                case 'unlisted':
                    return 'unlisted';
                case 'private':
                    return 'private';
                case 'direct':
                    return 'direct';
                case 'local':
                    return 'local';
            }
        };
        Converter.encodeVisibility = (v) => {
            switch (v) {
                case 'public':
                    return 'public';
                case 'unlisted':
                    return 'unlisted';
                case 'private':
                    return 'private';
                case 'direct':
                    return 'direct';
                case 'local':
                    return 'local';
            }
        };
        Converter.account = (a) => {
            return {
                id: a.id,
                username: a.username,
                acct: a.acct,
                display_name: a.display_name,
                locked: a.locked,
                discoverable: a.discoverable,
                group: null,
                noindex: a.noindex,
                suspended: a.suspended,
                limited: a.limited,
                created_at: a.created_at,
                followers_count: a.followers_count,
                following_count: a.following_count,
                statuses_count: a.statuses_count,
                note: a.note,
                url: a.url,
                avatar: a.avatar,
                avatar_static: a.avatar_static,
                header: a.header,
                header_static: a.header_static,
                emojis: a.emojis.map(e => Converter.emoji(e)),
                moved: a.moved ? Converter.account(a.moved) : null,
                fields: a.fields,
                bot: a.bot,
                source: a.source
            };
        };
        Converter.activity = (a) => a;
        Converter.announcement = (a) => ({
            id: a.id,
            content: a.content,
            starts_at: a.starts_at,
            ends_at: a.ends_at,
            published: a.published,
            all_day: a.all_day,
            published_at: a.published_at,
            updated_at: a.updated_at,
            read: null,
            mentions: a.mentions,
            statuses: a.statuses,
            tags: a.tags,
            emojis: a.emojis,
            reactions: a.reactions
        });
        Converter.application = (a) => a;
        Converter.attachment = (a) => a;
        Converter.async_attachment = (a) => {
            if (a.url) {
                return {
                    id: a.id,
                    type: a.type,
                    url: a.url,
                    remote_url: a.remote_url,
                    preview_url: a.preview_url,
                    text_url: a.text_url,
                    meta: a.meta,
                    description: a.description,
                    blurhash: a.blurhash
                };
            }
            else {
                return a;
            }
        };
        Converter.card = (c) => ({
            url: c.url,
            title: c.title,
            description: c.description,
            type: c.type,
            image: c.image,
            author_name: null,
            author_url: null,
            provider_name: c.provider_name,
            provider_url: c.provider_url,
            html: null,
            width: null,
            height: null,
            embed_url: null,
            blurhash: null
        });
        Converter.context = (c) => ({
            ancestors: Array.isArray(c.ancestors) ? c.ancestors.map(a => Converter.status(a)) : [],
            descendants: Array.isArray(c.descendants) ? c.descendants.map(d => Converter.status(d)) : []
        });
        Converter.conversation = (c) => ({
            id: c.id,
            accounts: Array.isArray(c.accounts) ? c.accounts.map(a => Converter.account(a)) : [],
            last_status: c.last_status ? Converter.status(c.last_status) : null,
            unread: c.unread
        });
        Converter.emoji = (e) => ({
            shortcode: e.shortcode,
            static_url: e.static_url,
            url: e.url,
            visible_in_picker: e.visible_in_picker
        });
        Converter.featured_tag = (f) => f;
        Converter.field = (f) => f;
        Converter.filter = (f) => f;
        Converter.history = (h) => h;
        Converter.identity_proof = (i) => i;
        Converter.instance = (i) => ({
            uri: i.uri,
            title: i.title,
            description: i.description,
            email: i.email,
            version: i.version,
            thumbnail: i.thumbnail,
            urls: Converter.urls(i.urls),
            stats: Converter.stats(i.stats),
            languages: i.languages,
            registrations: i.registrations,
            approval_required: i.approval_required,
            configuration: {
                statuses: {
                    max_characters: i.max_toot_chars,
                    max_media_attachments: i.max_media_attachments
                },
                polls: {
                    max_options: i.poll_limits.max_options,
                    max_characters_per_option: i.poll_limits.max_option_chars,
                    min_expiration: i.poll_limits.min_expiration,
                    max_expiration: i.poll_limits.max_expiration
                }
            }
        });
        Converter.list = (l) => ({
            id: l.id,
            title: l.title,
            replies_policy: null
        });
        Converter.marker = (m) => {
            if (m.notifications) {
                const mm = m;
                return {
                    notifications: {
                        last_read_id: mm.notifications.last_read_id,
                        version: mm.notifications.version,
                        updated_at: mm.notifications.updated_at,
                        unread_count: mm.notifications.pleroma.unread_count
                    }
                };
            }
            else {
                return {};
            }
        };
        Converter.mention = (m) => m;
        Converter.notification = (n) => {
            const notificationType = Converter.decodeNotificationType(n.type);
            if (notificationType instanceof UnknownNotificationTypeError)
                return notificationType;
            if (n.status && n.emoji) {
                const r = Converter.mapReaction(n);
                return {
                    id: n.id,
                    account: Converter.account(n.account),
                    created_at: n.created_at,
                    status: Converter.status(n.status),
                    reaction: r ? Converter.reaction(r) : undefined,
                    type: notificationType
                };
            }
            else if (n.status) {
                return {
                    id: n.id,
                    account: Converter.account(n.account),
                    created_at: n.created_at,
                    status: Converter.status(n.status),
                    type: notificationType
                };
            }
            else if (n.target) {
                return {
                    id: n.id,
                    account: Converter.account(n.account),
                    created_at: n.created_at,
                    target: Converter.account(n.target),
                    type: notificationType
                };
            }
            else {
                return {
                    id: n.id,
                    account: Converter.account(n.account),
                    created_at: n.created_at,
                    type: notificationType
                };
            }
        };
        Converter.mapReaction = (n) => {
            if (!n.emoji)
                return undefined;
            const name = n.emoji.replace(/:/g, '');
            let r = {
                count: 1,
                me: false,
                name: name
            };
            if (n.emoji_url) {
                r = Object.assign({}, r, {
                    url: n.emoji_url
                });
            }
            return r;
        };
        Converter.poll = (p) => p;
        Converter.pollOption = (p) => p;
        Converter.preferences = (p) => p;
        Converter.push_subscription = (p) => p;
        Converter.reaction = (r) => {
            let p = {
                count: r.count,
                me: r.me,
                name: r.name
            };
            if (r.url) {
                p = Object.assign({}, p, {
                    url: r.url,
                    static_url: r.url
                });
            }
            if (r.accounts) {
                p = Object.assign({}, p, {
                    accounts: r.accounts.map(a => Converter.account(a))
                });
            }
            if (r.account_ids) {
                p = Object.assign({}, p, {
                    accounts: r.account_ids
                });
            }
            return p;
        };
        Converter.relationship = (r) => ({
            id: r.id,
            following: r.following,
            followed_by: r.followed_by,
            blocking: r.blocking,
            blocked_by: r.blocked_by,
            muting: r.muting,
            muting_notifications: r.muting_notifications,
            requested: r.requested,
            domain_blocking: r.domain_blocking,
            showing_reblogs: r.showing_reblogs,
            endorsed: r.endorsed,
            notifying: r.notifying,
            note: r.note
        });
        Converter.report = (r) => ({
            id: r.id,
            action_taken: r.action_taken,
            action_taken_at: null,
            category: null,
            comment: null,
            forwarded: null,
            status_ids: null,
            rule_ids: null
        });
        Converter.results = (r) => ({
            accounts: Array.isArray(r.accounts) ? r.accounts.map(a => Converter.account(a)) : [],
            statuses: Array.isArray(r.statuses) ? r.statuses.map(s => Converter.status(s)) : [],
            hashtags: Array.isArray(r.hashtags) ? r.hashtags.map(h => Converter.tag(h)) : []
        });
        Converter.scheduled_status = (s) => ({
            id: s.id,
            scheduled_at: s.scheduled_at,
            params: Converter.status_params(s.params),
            media_attachments: Array.isArray(s.media_attachments) ? s.media_attachments.map(m => Converter.attachment(m)) : null
        });
        Converter.source = (s) => s;
        Converter.stats = (s) => s;
        Converter.status = (s) => ({
            id: s.id,
            uri: s.uri,
            url: s.url,
            account: Converter.account(s.account),
            in_reply_to_id: s.in_reply_to_id,
            in_reply_to_account_id: s.in_reply_to_account_id,
            reblog: s.reblog ? Converter.status(s.reblog) : null,
            content: s.content,
            plain_content: s.pleroma.content?.['text/plain'] ? s.pleroma.content['text/plain'] : null,
            created_at: s.created_at,
            edited_at: s.edited_at,
            emojis: Array.isArray(s.emojis) ? s.emojis.map(e => Converter.emoji(e)) : [],
            replies_count: s.replies_count,
            reblogs_count: s.reblogs_count,
            favourites_count: s.favourites_count,
            reblogged: s.reblogged,
            favourited: s.favourited,
            muted: s.muted,
            sensitive: s.sensitive,
            spoiler_text: s.spoiler_text,
            visibility: Converter.visibility(s.visibility),
            media_attachments: Array.isArray(s.media_attachments) ? s.media_attachments.map(m => Converter.attachment(m)) : [],
            mentions: Array.isArray(s.mentions) ? s.mentions.map(m => Converter.mention(m)) : [],
            tags: s.tags,
            card: s.card ? Converter.card(s.card) : null,
            poll: s.poll ? Converter.poll(s.poll) : null,
            application: s.application ? Converter.application(s.application) : null,
            language: s.language,
            pinned: s.pinned,
            emoji_reactions: Array.isArray(s.pleroma.emoji_reactions) ? s.pleroma.emoji_reactions.map(r => Converter.reaction(r)) : [],
            bookmarked: s.bookmarked ? s.bookmarked : false,
            quote: s.reblog !== null && s.reblog.content !== s.content
        });
        Converter.status_params = (s) => {
            return {
                text: s.text,
                in_reply_to_id: s.in_reply_to_id,
                media_ids: Array.isArray(s.media_ids) ? s.media_ids : null,
                sensitive: s.sensitive,
                spoiler_text: s.spoiler_text,
                visibility: s.visibility ? Converter.visibility(s.visibility) : null,
                scheduled_at: s.scheduled_at,
                application_id: null
            };
        };
        Converter.status_source = (s) => s;
        Converter.tag = (t) => t;
        Converter.token = (t) => t;
        Converter.urls = (u) => u;
    })(Converter = PleromaAPI.Converter || (PleromaAPI.Converter = {}));
    class Client {
        static DEFAULT_SCOPE = DEFAULT_SCOPE;
        static DEFAULT_URL = 'https://pleroma.io';
        static NO_REDIRECT = NO_REDIRECT;
        accessToken;
        baseUrl;
        userAgent;
        abortController;
        constructor(baseUrl, accessToken = null, userAgent = DEFAULT_UA) {
            this.accessToken = accessToken;
            this.baseUrl = baseUrl;
            this.userAgent = userAgent;
            this.abortController = new AbortController();
            axios.defaults.signal = this.abortController.signal;
        }
        async get(path, params = {}, headers = {}) {
            let options = {
                params: params,
                headers: headers
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .get(this.baseUrl + path, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async put(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .put(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async putForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .putForm(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async patch(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .patch(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async patchForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .patchForm(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async post(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios.post(this.baseUrl + path, params, options).then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async postForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios.postForm(this.baseUrl + path, params, options).then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async del(path, params = {}, headers = {}) {
            let options = {
                data: params,
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .delete(this.baseUrl + path, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        cancel() {
            return this.abortController.abort();
        }
        socket(url, stream, params) {
            if (!this.accessToken) {
                throw new Error('accessToken is required');
            }
            const streaming = new WebSocket(url, stream, params, this.accessToken, this.userAgent);
            streaming.start();
            return streaming;
        }
    }
    PleromaAPI.Client = Client;
})(PleromaAPI || (PleromaAPI = {}));
export default PleromaAPI;
