import axios from 'axios';
import objectAssignDeep from 'object-assign-deep';
import Streaming from './web_socket.js';
import { RequestCanceledError } from '../cancel.js';
import { NO_REDIRECT, DEFAULT_SCOPE, DEFAULT_UA } from '../default.js';
import NotificationType, { UnknownNotificationTypeError } from '../notification.js';
import MastodonNotificationType from './notification.js';
var MastodonAPI;
(function (MastodonAPI) {
    class Client {
        static DEFAULT_SCOPE = DEFAULT_SCOPE;
        static DEFAULT_URL = 'https://mastodon.social';
        static NO_REDIRECT = NO_REDIRECT;
        accessToken;
        baseUrl;
        userAgent;
        abortController;
        constructor(baseUrl, accessToken = null, userAgent = DEFAULT_UA) {
            this.accessToken = accessToken;
            this.baseUrl = baseUrl;
            this.userAgent = userAgent;
            this.abortController = new AbortController();
            axios.defaults.signal = this.abortController.signal;
        }
        async get(path, params = {}, headers = {}, pathIsFullyQualified = false) {
            let options = {
                params: params,
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .get((pathIsFullyQualified ? '' : this.baseUrl) + path, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async put(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .put(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async putForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .putForm(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async patch(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .patch(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async patchForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .patchForm(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async post(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios.post(this.baseUrl + path, params, options).then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async postForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios.postForm(this.baseUrl + path, params, options).then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async del(path, params = {}, headers = {}) {
            let options = {
                data: params,
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .delete(this.baseUrl + path, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        cancel() {
            return this.abortController.abort();
        }
        socket(url, stream, params) {
            if (!this.accessToken) {
                throw new Error('accessToken is required');
            }
            const streaming = new Streaming(url, stream, params, this.accessToken, this.userAgent);
            streaming.start();
            return streaming;
        }
    }
    MastodonAPI.Client = Client;
    let Converter;
    (function (Converter) {
        Converter.encodeNotificationType = (t) => {
            switch (t) {
                case NotificationType.Follow:
                    return MastodonNotificationType.Follow;
                case NotificationType.Favourite:
                    return MastodonNotificationType.Favourite;
                case NotificationType.Reblog:
                    return MastodonNotificationType.Reblog;
                case NotificationType.Mention:
                    return MastodonNotificationType.Mention;
                case NotificationType.FollowRequest:
                    return MastodonNotificationType.FollowRequest;
                case NotificationType.Status:
                    return MastodonNotificationType.Status;
                case NotificationType.PollExpired:
                    return MastodonNotificationType.Poll;
                case NotificationType.Update:
                    return MastodonNotificationType.Update;
                case NotificationType.AdminSignup:
                    return MastodonNotificationType.AdminSignup;
                case NotificationType.AdminReport:
                    return MastodonNotificationType.AdminReport;
                default:
                    return new UnknownNotificationTypeError();
            }
        };
        Converter.decodeNotificationType = (t) => {
            switch (t) {
                case MastodonNotificationType.Follow:
                    return NotificationType.Follow;
                case MastodonNotificationType.Favourite:
                    return NotificationType.Favourite;
                case MastodonNotificationType.Mention:
                    return NotificationType.Mention;
                case MastodonNotificationType.Reblog:
                    return NotificationType.Reblog;
                case MastodonNotificationType.FollowRequest:
                    return NotificationType.FollowRequest;
                case MastodonNotificationType.Status:
                    return NotificationType.Status;
                case MastodonNotificationType.Poll:
                    return NotificationType.PollExpired;
                case MastodonNotificationType.Update:
                    return NotificationType.Update;
                case MastodonNotificationType.AdminSignup:
                    return NotificationType.AdminSignup;
                case MastodonNotificationType.AdminReport:
                    return NotificationType.AdminReport;
                default:
                    return new UnknownNotificationTypeError();
            }
        };
        Converter.visibility = (v) => {
            switch (v) {
                case 'public':
                    return 'public';
                case 'unlisted':
                    return 'unlisted';
                case 'private':
                    return 'private';
                case 'direct':
                    return 'direct';
            }
        };
        Converter.encodeVisibility = (v) => {
            switch (v) {
                case 'public':
                    return 'public';
                case 'unlisted':
                    return 'unlisted';
                case 'private':
                    return 'private';
                case 'direct':
                    return 'direct';
                case 'local':
                    return 'public';
            }
        };
        Converter.account = (a) => a;
        Converter.activity = (a) => a;
        Converter.announcement = (a) => a;
        Converter.application = (a) => a;
        Converter.attachment = (a) => a;
        Converter.async_attachment = (a) => {
            if (a.url) {
                return {
                    id: a.id,
                    type: a.type,
                    url: a.url,
                    remote_url: a.remote_url,
                    preview_url: a.preview_url,
                    text_url: a.text_url,
                    meta: a.meta,
                    description: a.description,
                    blurhash: a.blurhash
                };
            }
            else {
                return a;
            }
        };
        Converter.card = (c) => c;
        Converter.context = (c) => ({
            ancestors: Array.isArray(c.ancestors) ? c.ancestors.map(a => Converter.status(a)) : [],
            descendants: Array.isArray(c.descendants) ? c.descendants.map(d => Converter.status(d)) : []
        });
        Converter.conversation = (c) => ({
            id: c.id,
            accounts: Array.isArray(c.accounts) ? c.accounts.map(a => Converter.account(a)) : [],
            last_status: c.last_status ? Converter.status(c.last_status) : null,
            unread: c.unread
        });
        Converter.emoji = (e) => e;
        Converter.featured_tag = (e) => e;
        Converter.field = (f) => f;
        Converter.filter = (f) => f;
        Converter.history = (h) => h;
        Converter.identity_proof = (i) => i;
        Converter.instance = (i) => i;
        Converter.list = (l) => l;
        Converter.marker = (m) => m;
        Converter.mention = (m) => m;
        Converter.notification = (n) => {
            const notificationType = Converter.decodeNotificationType(n.type);
            if (notificationType instanceof UnknownNotificationTypeError)
                return notificationType;
            if (n.status) {
                return {
                    account: Converter.account(n.account),
                    created_at: n.created_at,
                    id: n.id,
                    status: Converter.status(n.status),
                    type: notificationType
                };
            }
            else {
                return {
                    account: Converter.account(n.account),
                    created_at: n.created_at,
                    id: n.id,
                    type: notificationType
                };
            }
        };
        Converter.poll = (p) => p;
        Converter.poll_option = (p) => p;
        Converter.preferences = (p) => p;
        Converter.push_subscription = (p) => p;
        Converter.relationship = (r) => r;
        Converter.report = (r) => r;
        Converter.results = (r) => ({
            accounts: Array.isArray(r.accounts) ? r.accounts.map(a => Converter.account(a)) : [],
            statuses: Array.isArray(r.statuses) ? r.statuses.map(s => Converter.status(s)) : [],
            hashtags: Array.isArray(r.hashtags) ? r.hashtags.map(h => Converter.tag(h)) : []
        });
        Converter.scheduled_status = (s) => s;
        Converter.source = (s) => s;
        Converter.stats = (s) => s;
        Converter.status = (s) => ({
            id: s.id,
            uri: s.uri,
            url: s.url,
            account: Converter.account(s.account),
            in_reply_to_id: s.in_reply_to_id,
            in_reply_to_account_id: s.in_reply_to_account_id,
            reblog: s.reblog ? Converter.status(s.reblog) : s.quote ? Converter.status(s.quote) : null,
            content: s.content,
            plain_content: null,
            created_at: s.created_at,
            edited_at: s.edited_at,
            emojis: Array.isArray(s.emojis) ? s.emojis.map(e => Converter.emoji(e)) : [],
            replies_count: s.replies_count,
            reblogs_count: s.reblogs_count,
            favourites_count: s.favourites_count,
            reblogged: s.reblogged,
            favourited: s.favourited,
            muted: s.muted,
            sensitive: s.sensitive,
            spoiler_text: s.spoiler_text,
            visibility: Converter.visibility(s.visibility),
            media_attachments: Array.isArray(s.media_attachments) ? s.media_attachments.map(m => Converter.attachment(m)) : [],
            mentions: Array.isArray(s.mentions) ? s.mentions.map(m => Converter.mention(m)) : [],
            tags: s.tags,
            card: s.card ? Converter.card(s.card) : null,
            poll: s.poll ? Converter.poll(s.poll) : null,
            application: s.application ? Converter.application(s.application) : null,
            language: s.language,
            pinned: s.pinned,
            emoji_reactions: [],
            bookmarked: s.bookmarked ? s.bookmarked : false,
            quote: s.quote !== undefined && s.quote !== null
        });
        Converter.status_params = (s) => s;
        Converter.status_source = (s) => s;
        Converter.tag = (t) => t;
        Converter.token = (t) => t;
        Converter.urls = (u) => u;
    })(Converter = MastodonAPI.Converter || (MastodonAPI.Converter = {}));
})(MastodonAPI || (MastodonAPI = {}));
export default MastodonAPI;
