import axios from 'axios';
import objectAssignDeep from 'object-assign-deep';
import Streaming from './web_socket.js';
import { RequestCanceledError } from '../cancel.js';
import { NO_REDIRECT, DEFAULT_SCOPE, DEFAULT_UA } from '../default.js';
import NotificationType, { UnknownNotificationTypeError } from '../notification.js';
import GotosocialNotificationType from './notification.js';
var GotosocialAPI;
(function (GotosocialAPI) {
    class Client {
        static DEFAULT_SCOPE = DEFAULT_SCOPE;
        static DEFAULT_URL = 'https://mastodon.social';
        static NO_REDIRECT = NO_REDIRECT;
        accessToken;
        baseUrl;
        userAgent;
        abortController;
        constructor(baseUrl, accessToken = null, userAgent = DEFAULT_UA) {
            this.accessToken = accessToken;
            this.baseUrl = baseUrl;
            this.userAgent = userAgent;
            this.abortController = new AbortController();
            axios.defaults.signal = this.abortController.signal;
        }
        async get(path, params = {}, headers = {}, pathIsFullyQualified = false) {
            let options = {
                params: params,
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .get((pathIsFullyQualified ? '' : this.baseUrl) + path, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async put(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .put(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async putForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .putForm(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async patch(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .patch(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async patchForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .patchForm(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async post(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios.post(this.baseUrl + path, params, options).then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async postForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios.postForm(this.baseUrl + path, params, options).then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async del(path, params = {}, headers = {}) {
            let options = {
                data: params,
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .delete(this.baseUrl + path, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        cancel() {
            return this.abortController.abort();
        }
        socket(url, stream, params) {
            if (!this.accessToken) {
                throw new Error('accessToken is required');
            }
            const streaming = new Streaming(url, stream, params, this.accessToken, this.userAgent);
            streaming.start();
            return streaming;
        }
    }
    GotosocialAPI.Client = Client;
    let Converter;
    (function (Converter) {
        Converter.encodeNotificationType = (t) => {
            switch (t) {
                case NotificationType.Follow:
                    return GotosocialNotificationType.Follow;
                case NotificationType.Favourite:
                    return GotosocialNotificationType.Favourite;
                case NotificationType.Reblog:
                    return GotosocialNotificationType.Reblog;
                case NotificationType.Mention:
                    return GotosocialNotificationType.Mention;
                case NotificationType.FollowRequest:
                    return GotosocialNotificationType.FollowRequest;
                case NotificationType.Status:
                    return GotosocialNotificationType.Status;
                case NotificationType.PollExpired:
                    return GotosocialNotificationType.Poll;
                default:
                    return new UnknownNotificationTypeError();
            }
        };
        Converter.decodeNotificationType = (t) => {
            switch (t) {
                case GotosocialNotificationType.Follow:
                    return NotificationType.Follow;
                case GotosocialNotificationType.Favourite:
                    return NotificationType.Favourite;
                case GotosocialNotificationType.Mention:
                    return NotificationType.Mention;
                case GotosocialNotificationType.Reblog:
                    return NotificationType.Reblog;
                case GotosocialNotificationType.FollowRequest:
                    return NotificationType.FollowRequest;
                case GotosocialNotificationType.Status:
                    return NotificationType.Status;
                case GotosocialNotificationType.Poll:
                    return NotificationType.PollExpired;
                default:
                    return new UnknownNotificationTypeError();
            }
        };
        Converter.visibility = (v) => {
            switch (v) {
                case 'public':
                    return 'public';
                case 'unlisted':
                    return 'unlisted';
                case 'private':
                    return 'private';
                case 'direct':
                    return 'direct';
                case 'local':
                    return 'local';
            }
        };
        Converter.encodeVisibility = (v) => {
            switch (v) {
                case 'public':
                    return 'public';
                case 'unlisted':
                    return 'unlisted';
                case 'private':
                    return 'private';
                case 'direct':
                    return 'direct';
                case 'local':
                    return 'local';
            }
        };
        Converter.account = (a) => ({
            id: a.id,
            username: a.username,
            acct: a.acct,
            display_name: a.display_name,
            locked: a.locked,
            discoverable: a.discoverable,
            group: null,
            noindex: null,
            suspended: a.suspended,
            limited: null,
            created_at: a.created_at,
            followers_count: a.followers_count,
            following_count: a.following_count,
            statuses_count: a.statuses_count,
            note: a.note,
            url: a.url,
            avatar: a.avatar,
            avatar_static: a.avatar_static,
            header: a.header,
            header_static: a.header_static,
            emojis: Array.isArray(a.emojis) ? a.emojis.map(e => Converter.emoji(e)) : [],
            moved: null,
            fields: Array.isArray(a.fields) ? a.fields.map(f => Converter.field(f)) : [],
            bot: a.bot,
            source: a.source ? Converter.source(a.source) : undefined,
            mute_expires_at: a.mute_expires_at,
            role: a.role ? a.role : undefined
        });
        Converter.application = (a) => a;
        Converter.attachment = (a) => ({
            id: a.id,
            type: a.type,
            url: a.url ? a.url : '',
            remote_url: a.remote_url,
            preview_url: a.preview_url,
            text_url: a.text_url,
            meta: a.meta,
            description: a.description,
            blurhash: a.blurhash
        });
        Converter.async_attachment = (a) => {
            if (a.url) {
                return {
                    id: a.id,
                    type: a.type,
                    url: a.url ? a.url : '',
                    remote_url: a.remote_url,
                    preview_url: a.preview_url,
                    text_url: a.text_url,
                    meta: a.meta,
                    description: a.description,
                    blurhash: a.blurhash
                };
            }
            else {
                return a;
            }
        };
        Converter.card = (c) => ({
            url: c.url,
            title: c.title,
            description: c.description,
            image: c.image,
            type: c.type,
            author_name: c.author_name,
            author_url: c.author_url,
            provider_name: c.provider_name,
            provider_url: c.provider_url,
            html: c.html,
            width: c.width,
            height: c.height,
            embed_url: null,
            blurhash: null
        });
        Converter.context = (c) => ({
            ancestors: Array.isArray(c.ancestors) ? c.ancestors.map(a => Converter.status(a)) : [],
            descendants: Array.isArray(c.descendants) ? c.descendants.map(d => Converter.status(d)) : []
        });
        Converter.emoji = (e) => e;
        Converter.field = (f) => f;
        Converter.instance = (i) => ({
            uri: i.uri,
            title: i.title,
            description: i.description,
            email: i.email,
            version: i.version,
            thumbnail: i.thumbnail,
            urls: Converter.urls(i.urls),
            stats: Converter.stats(i.stats),
            languages: i.languages,
            registrations: i.registrations,
            approval_required: i.approval_required,
            invites_enabled: i.invites_enabled,
            contact_account: i.contact_account ? Converter.account(i.contact_account) : undefined,
            configuration: i.configuration,
            rules: []
        });
        Converter.list = (l) => ({
            id: l.id,
            title: l.title,
            replies_policy: l.replies_policy ? l.replies_policy : null
        });
        Converter.marker = (m) => m;
        Converter.mention = (m) => m;
        Converter.notification = (n) => {
            const notificationType = Converter.decodeNotificationType(n.type);
            if (notificationType instanceof UnknownNotificationTypeError)
                return notificationType;
            if (n.status) {
                return {
                    account: Converter.account(n.account),
                    created_at: n.created_at,
                    id: n.id,
                    status: Converter.status(n.status),
                    type: notificationType
                };
            }
            else {
                return {
                    account: Converter.account(n.account),
                    created_at: n.created_at,
                    id: n.id,
                    type: notificationType
                };
            }
        };
        Converter.poll = (p) => p;
        Converter.poll_option = (p) => p;
        Converter.preferences = (p) => p;
        Converter.relationship = (r) => r;
        Converter.report = (r) => ({
            id: r.id,
            action_taken: r.action_taken,
            action_taken_at: r.action_taken_at,
            status_ids: r.status_ids,
            category: r.category,
            comment: r.comment,
            forwarded: r.forwarded,
            rule_ids: r.rule_ids,
            target_account: Converter.account(r.target_account)
        });
        Converter.results = (r) => ({
            accounts: Array.isArray(r.accounts) ? r.accounts.map(a => Converter.account(a)) : [],
            statuses: Array.isArray(r.statuses) ? r.statuses.map(s => Converter.status(s)) : [],
            hashtags: Array.isArray(r.hashtags) ? r.hashtags.map(h => Converter.tag(h)) : []
        });
        Converter.scheduled_status = (s) => ({
            id: s.id,
            media_attachments: Array.isArray(s.media_attachments) ? s.media_attachments.map(m => Converter.attachment(m)) : [],
            params: Converter.status_params(s.params),
            scheduled_at: s.scheduled_at
        });
        Converter.source = (s) => ({
            privacy: s.privacy,
            sensitive: s.sensitive,
            language: s.language,
            note: s.note,
            fields: Array.isArray(s.fields) ? s.fields.map(f => Converter.field(f)) : []
        });
        Converter.stats = (s) => s;
        Converter.status = (s) => ({
            id: s.id,
            uri: s.uri,
            url: s.url,
            account: Converter.account(s.account),
            in_reply_to_id: s.in_reply_to_id,
            in_reply_to_account_id: s.in_reply_to_account_id,
            reblog: s.reblog ? Converter.status(s.reblog) : null,
            content: s.content,
            plain_content: null,
            created_at: s.created_at,
            edited_at: null,
            emojis: Array.isArray(s.emojis) ? s.emojis.map(e => Converter.emoji(e)) : [],
            replies_count: s.replies_count,
            reblogs_count: s.reblogs_count,
            favourites_count: s.favourites_count,
            reblogged: s.reblogged,
            favourited: s.favourited,
            muted: s.muted,
            sensitive: s.sensitive,
            spoiler_text: s.spoiler_text,
            visibility: Converter.visibility(s.visibility),
            media_attachments: Array.isArray(s.media_attachments) ? s.media_attachments.map(m => Converter.attachment(m)) : [],
            mentions: Array.isArray(s.mentions) ? s.mentions.map(m => Converter.mention(m)) : [],
            tags: s.tags,
            card: s.card ? Converter.card(s.card) : null,
            poll: s.poll ? Converter.poll(s.poll) : null,
            application: s.application ? Converter.application(s.application) : null,
            language: s.language,
            pinned: s.pinned,
            emoji_reactions: [],
            bookmarked: s.bookmarked ? s.bookmarked : false,
            quote: false
        });
        Converter.status_params = (s) => s;
        Converter.tag = (t) => ({
            name: t.name,
            url: t.url,
            history: []
        });
        Converter.token = (t) => t;
        Converter.urls = (u) => u;
        Converter.filter = (f) => f;
        Converter.status_source = (s) => s;
    })(Converter = GotosocialAPI.Converter || (GotosocialAPI.Converter = {}));
})(GotosocialAPI || (GotosocialAPI = {}));
export default GotosocialAPI;
