import axios from 'axios';
import objectAssignDeep from 'object-assign-deep';
import WebSocket from './web_socket.js';
import { RequestCanceledError } from '../cancel.js';
import { NO_REDIRECT, DEFAULT_SCOPE, DEFAULT_UA } from '../default.js';
import NotificationType, { UnknownNotificationTypeError } from '../notification.js';
import FriendicaNotificationType from './notification.js';
var FriendicaAPI;
(function (FriendicaAPI) {
    class Client {
        static DEFAULT_SCOPE = DEFAULT_SCOPE;
        static DEFAULT_URL = 'https://mastodon.social';
        static NO_REDIRECT = NO_REDIRECT;
        accessToken;
        baseUrl;
        userAgent;
        abortController;
        constructor(baseUrl, accessToken = null, userAgent = DEFAULT_UA) {
            this.accessToken = accessToken;
            this.baseUrl = baseUrl;
            this.userAgent = userAgent;
            this.abortController = new AbortController();
            axios.defaults.signal = this.abortController.signal;
        }
        async get(path, params = {}, headers = {}, pathIsFullyQualified = false) {
            let options = {
                params: params,
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .get((pathIsFullyQualified ? '' : this.baseUrl) + path, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async put(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .put(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async putForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .putForm(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async patch(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .patch(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async patchForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .patchForm(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async post(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios.post(this.baseUrl + path, params, options).then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async postForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios.postForm(this.baseUrl + path, params, options).then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async del(path, params = {}, headers = {}) {
            let options = {
                data: params,
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .delete(this.baseUrl + path, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        cancel() {
            return this.abortController.abort();
        }
        socket(url, stream, params) {
            if (!this.accessToken) {
                throw new Error('accessToken is required');
            }
            const streaming = new WebSocket(url, stream, params, this.accessToken, this.userAgent);
            streaming.start();
            return streaming;
        }
    }
    FriendicaAPI.Client = Client;
    let Converter;
    (function (Converter) {
        Converter.encodeNotificationType = (t) => {
            switch (t) {
                case NotificationType.Follow:
                    return FriendicaNotificationType.Follow;
                case NotificationType.Favourite:
                    return FriendicaNotificationType.Favourite;
                case NotificationType.Reblog:
                    return FriendicaNotificationType.Reblog;
                case NotificationType.Mention:
                    return FriendicaNotificationType.Mention;
                case NotificationType.FollowRequest:
                    return FriendicaNotificationType.FollowRequest;
                case NotificationType.Status:
                    return FriendicaNotificationType.Status;
                case NotificationType.PollExpired:
                    return FriendicaNotificationType.Poll;
                case NotificationType.Update:
                    return FriendicaNotificationType.Update;
                default:
                    return new UnknownNotificationTypeError();
            }
        };
        Converter.decodeNotificationType = (t) => {
            switch (t) {
                case FriendicaNotificationType.Follow:
                    return NotificationType.Follow;
                case FriendicaNotificationType.Favourite:
                    return NotificationType.Favourite;
                case FriendicaNotificationType.Mention:
                    return NotificationType.Mention;
                case FriendicaNotificationType.Reblog:
                    return NotificationType.Reblog;
                case FriendicaNotificationType.FollowRequest:
                    return NotificationType.FollowRequest;
                case FriendicaNotificationType.Status:
                    return NotificationType.Status;
                case FriendicaNotificationType.Poll:
                    return NotificationType.PollExpired;
                case FriendicaNotificationType.Update:
                    return NotificationType.Update;
                default:
                    return new UnknownNotificationTypeError();
            }
        };
        Converter.visibility = (v) => {
            switch (v) {
                case 'public':
                    return 'public';
                case 'unlisted':
                    return 'unlisted';
                case 'private':
                    return 'private';
            }
        };
        Converter.encodeVisibility = (v) => {
            switch (v) {
                case 'public':
                    return 'public';
                case 'unlisted':
                    return 'unlisted';
                case 'direct':
                case 'private':
                    return 'private';
                default:
                    return 'public';
            }
        };
        Converter.account = (a) => ({
            id: a.id,
            username: a.username,
            acct: a.acct,
            display_name: a.display_name,
            locked: a.locked,
            discoverable: a.discoverable,
            group: a.group,
            noindex: null,
            suspended: null,
            limited: null,
            created_at: a.created_at,
            followers_count: a.followers_count,
            following_count: a.following_count,
            statuses_count: a.statuses_count,
            note: a.note,
            url: a.url,
            avatar: a.avatar,
            avatar_static: a.avatar_static,
            header: a.header,
            header_static: a.header_static,
            emojis: a.emojis.map(e => Converter.emoji(e)),
            moved: a.moved ? Converter.account(a.moved) : null,
            fields: a.fields.map(f => Converter.field(f)),
            bot: a.bot,
            source: a.source ? Converter.source(a.source) : undefined
        });
        Converter.activity = (a) => a;
        Converter.application = (a) => a;
        Converter.attachment = (a) => a;
        Converter.async_attachment = (a) => {
            if (a.url) {
                return {
                    id: a.id,
                    type: a.type,
                    url: a.url,
                    remote_url: a.remote_url,
                    preview_url: a.preview_url,
                    text_url: a.text_url,
                    meta: a.meta,
                    description: a.description,
                    blurhash: a.blurhash
                };
            }
            else {
                return a;
            }
        };
        Converter.card = (c) => ({
            url: c.url,
            title: c.title,
            description: c.description,
            type: c.type,
            image: c.image,
            author_name: c.author_name,
            author_url: c.author_url,
            provider_name: c.provider_name,
            provider_url: c.provider_url,
            html: c.html,
            width: c.width,
            height: c.height,
            embed_url: null,
            blurhash: c.blurhash
        });
        Converter.context = (c) => ({
            ancestors: Array.isArray(c.ancestors) ? c.ancestors.map(a => Converter.status(a)) : [],
            descendants: Array.isArray(c.descendants) ? c.descendants.map(d => Converter.status(d)) : []
        });
        Converter.conversation = (c) => ({
            id: c.id,
            accounts: Array.isArray(c.accounts) ? c.accounts.map(a => Converter.account(a)) : [],
            last_status: c.last_status ? Converter.status(c.last_status) : null,
            unread: c.unread
        });
        Converter.emoji = (e) => ({
            shortcode: e.shortcode,
            static_url: e.static_url,
            url: e.url,
            visible_in_picker: e.visible_in_picker
        });
        Converter.featured_tag = (e) => e;
        Converter.field = (f) => f;
        Converter.filter = (f) => f;
        Converter.follow_request = (f) => ({
            id: f.id,
            username: f.username,
            acct: f.acct,
            display_name: f.display_name,
            locked: f.locked,
            bot: f.bot,
            discoverable: f.discoverable,
            group: f.group,
            created_at: f.created_at,
            note: f.note,
            url: f.url,
            avatar: f.avatar,
            avatar_static: f.avatar_static,
            header: f.header,
            header_static: f.header_static,
            followers_count: f.followers_count,
            following_count: f.following_count,
            statuses_count: f.statuses_count,
            emojis: f.emojis.map(e => Converter.emoji(e)),
            fields: f.fields.map(f => Converter.field(f))
        });
        Converter.history = (h) => h;
        Converter.identity_proof = (i) => i;
        Converter.instance = (i) => {
            return {
                uri: i.uri,
                title: i.title,
                description: i.description,
                email: i.email,
                version: i.version,
                thumbnail: i.thumbnail,
                urls: i.urls ? Converter.urls(i.urls) : null,
                stats: Converter.stats(i.stats),
                languages: i.languages,
                registrations: i.registrations,
                approval_required: i.approval_required,
                invites_enabled: i.invites_enabled,
                configuration: {
                    statuses: {
                        max_characters: i.max_toot_chars
                    }
                },
                contact_account: Converter.account(i.contact_account),
                rules: i.rules
            };
        };
        Converter.list = (l) => l;
        Converter.marker = (m) => m;
        Converter.mention = (m) => m;
        Converter.notification = (n) => {
            const notificationType = Converter.decodeNotificationType(n.type);
            if (notificationType instanceof UnknownNotificationTypeError)
                return notificationType;
            if (n.status) {
                return {
                    account: Converter.account(n.account),
                    created_at: n.created_at,
                    id: n.id,
                    status: Converter.status(n.status),
                    type: notificationType
                };
            }
            else {
                return {
                    account: Converter.account(n.account),
                    created_at: n.created_at,
                    id: n.id,
                    type: notificationType
                };
            }
        };
        Converter.poll = (p) => p;
        Converter.poll_option = (p) => p;
        Converter.preferences = (p) => p;
        Converter.push_subscription = (p) => p;
        Converter.relationship = (r) => r;
        Converter.report = (r) => ({
            id: r.id,
            action_taken: r.action_taken,
            action_taken_at: null,
            category: r.category,
            comment: r.comment,
            forwarded: r.forwarded,
            status_ids: r.status_ids,
            rule_ids: r.rule_ids,
            target_account: Converter.account(r.target_account)
        });
        Converter.results = (r) => ({
            accounts: Array.isArray(r.accounts) ? r.accounts.map(a => Converter.account(a)) : [],
            statuses: Array.isArray(r.statuses) ? r.statuses.map(s => Converter.status(s)) : [],
            hashtags: Array.isArray(r.hashtags) ? r.hashtags.map(h => Converter.tag(h)) : []
        });
        Converter.scheduled_status = (s) => {
            return {
                id: s.id,
                scheduled_at: s.scheduled_at,
                params: Converter.status_params(s.params),
                media_attachments: s.media_attachments ? s.media_attachments.map(a => Converter.attachment(a)) : null
            };
        };
        Converter.source = (s) => s;
        Converter.stats = (s) => s;
        Converter.status = (s) => ({
            id: s.id,
            uri: s.uri,
            url: s.url,
            account: Converter.account(s.account),
            in_reply_to_id: s.in_reply_to_id,
            in_reply_to_account_id: s.in_reply_to_account_id,
            reblog: s.reblog ? Converter.status(s.reblog) : s.quote ? Converter.status(s.quote) : null,
            content: s.content,
            plain_content: null,
            created_at: s.created_at,
            edited_at: null,
            emojis: Array.isArray(s.emojis) ? s.emojis.map(e => Converter.emoji(e)) : [],
            replies_count: s.replies_count,
            reblogs_count: s.reblogs_count,
            favourites_count: s.favourites_count,
            reblogged: s.reblogged,
            favourited: s.favourited,
            muted: s.muted,
            sensitive: s.sensitive,
            spoiler_text: s.spoiler_text,
            visibility: Converter.visibility(s.visibility),
            media_attachments: Array.isArray(s.media_attachments) ? s.media_attachments.map(m => Converter.attachment(m)) : [],
            mentions: Array.isArray(s.mentions) ? s.mentions.map(m => Converter.mention(m)) : [],
            tags: s.tags,
            card: s.card ? Converter.card(s.card) : null,
            poll: s.poll ? Converter.poll(s.poll) : null,
            application: s.application ? Converter.application(s.application) : null,
            language: s.language,
            pinned: s.pinned,
            emoji_reactions: [],
            bookmarked: s.bookmarked ? s.bookmarked : false,
            quote: false
        });
        Converter.status_params = (s) => {
            return {
                text: s.text,
                in_reply_to_id: s.in_reply_to_id,
                media_ids: s.media_ids,
                sensitive: s.sensitive,
                spoiler_text: s.spoiler_text,
                visibility: s.visibility ? Converter.visibility(s.visibility) : null,
                scheduled_at: s.scheduled_at,
                application_id: parseInt(s.application_id)
            };
        };
        Converter.status_source = (s) => s;
        Converter.tag = (t) => t;
        Converter.token = (t) => t;
        Converter.urls = (u) => u;
    })(Converter = FriendicaAPI.Converter || (FriendicaAPI.Converter = {}));
})(FriendicaAPI || (FriendicaAPI = {}));
export default FriendicaAPI;
