"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpActionDispatcherHookMastodon = void 0;
const change_case_1 = require("change-case");
const index_js_1 = require("../../utils/index.js");
const index_js_2 = require("../errors/index.js");
function isHttpActionType(actionType) {
    return ["fetch", "create", "update", "remove", "list"].includes(actionType);
}
function toHttpActionType(action) {
    if (isHttpActionType(action)) {
        return action;
    }
    switch (action) {
        case "lookup":
        case "verify_credentials": {
            return "fetch";
        }
        case "update_credentials": {
            return "update";
        }
        default: {
            return "create";
        }
    }
}
function inferEncoding(action, path) {
    if ((action === "create" && path === "/api/v1/accounts") ||
        (action === "update" && path === "/api/v1/accounts/update_credentials") ||
        (action === "create" && path === "/api/v1/email") ||
        (action === "create" && path === "/api/v1/featured_tag") ||
        (action === "create" && path === "/api/v1/media") ||
        (action === "create" && path === "/api/v2/media")) {
        return "multipart-form";
    }
    return "json";
}
async function waitForMediaAttachment(id, timeout, http) {
    let media;
    const signal = AbortSignal.timeout(timeout);
    while (!media) {
        if (signal.aborted) {
            throw new index_js_2.MastoTimeoutError(`Media processing timed out of ${timeout}ms`);
        }
        try {
            await (0, index_js_1.sleep)(1000);
            const processing = await http
                .get(`/api/v1/media/${id}`)
                .then((r) => r.data);
            if (processing.url) {
                media = processing;
            }
        }
        catch (error) {
            if (error instanceof index_js_2.MastoHttpError && error.statusCode === 404) {
                continue;
            }
            throw error;
        }
    }
    return media;
}
class HttpActionDispatcherHookMastodon {
    http;
    mediaTimeout;
    constructor(http, mediaTimeout = 1000 * 60) {
        this.http = http;
        this.mediaTimeout = mediaTimeout;
    }
    beforeDispatch(action) {
        const type = toHttpActionType(action.type);
        const path = isHttpActionType(action.type)
            ? action.path
            : action.path + "/" + (0, change_case_1.snakeCase)(action.type);
        const encoding = inferEncoding(type, path);
        const meta = { ...action.meta, encoding };
        return { ...action, type, path, meta };
    }
    dispatch(action) {
        if (action.type === "update" &&
            action.path === "/api/v1/accounts/update_credentials") {
            return this.http
                .patch(action.path, action.data, action.meta)
                .then((r) => r.data);
        }
        return false;
    }
    afterDispatch(action, result) {
        if (action.type === "create" && action.path === "/api/v2/media") {
            const media = result;
            if ((0, index_js_1.isRecord)(action.data) && action.data?.skipPolling === true) {
                return media;
            }
            return waitForMediaAttachment(media.id, this.mediaTimeout, this.http);
        }
        return result;
    }
}
exports.HttpActionDispatcherHookMastodon = HttpActionDispatcherHookMastodon;
