import { LexUserType, LexiconDoc, ValidationResult } from './types';
/**
 * A collection of compiled lexicons.
 */
export declare class Lexicons implements Iterable<LexiconDoc> {
    docs: Map<string, LexiconDoc>;
    defs: Map<string, LexUserType>;
    constructor(docs?: Iterable<LexiconDoc>);
    /**
     * @example clone a lexicon:
     * ```ts
     * const clone = new Lexicons(originalLexicon)
     * ```
     *
     * @example get docs array:
     * ```ts
     * const docs = Array.from(lexicons)
     * ```
     */
    [Symbol.iterator](): Iterator<LexiconDoc>;
    /**
     * Add a lexicon doc.
     */
    add(doc: LexiconDoc): void;
    /**
     * Remove a lexicon doc.
     */
    remove(uri: string): void;
    /**
     * Get a lexicon doc.
     */
    get(uri: string): LexiconDoc | undefined;
    /**
     * Get a definition.
     */
    getDef(uri: string): LexUserType | undefined;
    /**
     * Get a def, throw if not found. Throws on not found.
     */
    getDefOrThrow<T extends LexUserType['type'] = LexUserType['type']>(uri: string, types?: readonly T[]): Extract<LexUserType, {
        type: T;
    }>;
    /**
     * Validate a record or object.
     */
    validate(lexUri: string, value: unknown): ValidationResult;
    /**
     * Validate a record and throw on any error.
     */
    assertValidRecord(lexUri: string, value: unknown): unknown;
    /**
     * Validate xrpc query params and throw on any error.
     */
    assertValidXrpcParams(lexUri: string, value: unknown): Record<string, unknown> | undefined;
    /**
     * Validate xrpc input body and throw on any error.
     */
    assertValidXrpcInput(lexUri: string, value: unknown): unknown;
    /**
     * Validate xrpc output body and throw on any error.
     */
    assertValidXrpcOutput(lexUri: string, value: unknown): unknown;
    /**
     * Validate xrpc subscription message and throw on any error.
     */
    assertValidXrpcMessage<T = unknown>(lexUri: string, value: unknown): T;
    /**
     * Resolve a lex uri given a ref
     */
    resolveLexUri(lexUri: string, ref: string): string;
}
//# sourceMappingURL=lexicons.d.ts.map