"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.lexStringify = lexStringify;
exports.lexParse = lexParse;
exports.jsonToLex = jsonToLex;
exports.lexToJson = lexToJson;
const lex_data_1 = require("@atproto/lex-data");
const blob_js_1 = require("./blob.js");
const bytes_js_1 = require("./bytes.js");
const link_js_1 = require("./link.js");
function lexStringify(input) {
    // @NOTE Because of the way the "replacer" works in JSON.stringify, it's
    // simpler to convert Lex to JSON first rather than trying to do it
    // on-the-fly.
    return JSON.stringify(lexToJson(input));
}
function lexParse(input, options = { strict: false }) {
    return JSON.parse(input, function (key, value) {
        switch (typeof value) {
            case 'object':
                if (value === null)
                    return null;
                if (Array.isArray(value))
                    return value;
                return parseSpecialJsonObject(value, options) ?? value;
            case 'number':
                if (Number.isInteger(value))
                    return value;
                if (options.strict) {
                    throw new TypeError(`Invalid non-integer number: ${value}`);
                }
            // fallthrough
            default:
                return value;
        }
    });
}
function jsonToLex(value, options = { strict: false }) {
    switch (typeof value) {
        case 'object': {
            if (value === null)
                return null;
            if (Array.isArray(value))
                return jsonArrayToLex(value, options);
            return (parseSpecialJsonObject(value, options) ??
                jsonObjectToLexMap(value, options));
        }
        case 'number':
            if (Number.isInteger(value))
                return value;
            if (options.strict) {
                throw new TypeError(`Invalid non-integer number: ${value}`);
            }
        // fallthrough
        case 'boolean':
        case 'string':
            return value;
        default:
            throw new TypeError(`Invalid JSON value: ${typeof value}`);
    }
}
function jsonArrayToLex(input, options) {
    // Lazily copy value
    let copy;
    for (let i = 0; i < input.length; i++) {
        const inputItem = input[i];
        const item = jsonToLex(inputItem, options);
        if (item !== inputItem) {
            copy ?? (copy = Array.from(input));
            copy[i] = item;
        }
    }
    return copy ?? input;
}
function jsonObjectToLexMap(input, options) {
    // Lazily copy value
    let copy = undefined;
    for (const [key, jsonValue] of Object.entries(input)) {
        // Prevent prototype pollution
        if (key === '__proto__') {
            throw new TypeError('Invalid key: __proto__');
        }
        // Ignore (strip) undefined values
        if (jsonValue === undefined) {
            copy ?? (copy = { ...input });
            delete copy[key];
            continue;
        }
        const value = jsonToLex(jsonValue, options);
        if (value !== jsonValue) {
            copy ?? (copy = { ...input });
            copy[key] = value;
        }
    }
    return copy ?? input;
}
function lexToJson(value) {
    switch (typeof value) {
        case 'object':
            if (value === null) {
                return value;
            }
            else if (Array.isArray(value)) {
                return lexArrayToJson(value);
            }
            else if ((0, lex_data_1.isCid)(value)) {
                return (0, link_js_1.encodeLexLink)(value);
            }
            else if (value instanceof Uint8Array) {
                return (0, bytes_js_1.encodeLexBytes)(value);
            }
            else {
                return encodeLexMap(value);
            }
        case 'boolean':
        case 'string':
        case 'number':
            return value;
        default:
            throw new TypeError(`Invalid Lex value: ${typeof value}`);
    }
}
function lexArrayToJson(input) {
    // Lazily copy value
    let copy;
    for (let i = 0; i < input.length; i++) {
        const inputItem = input[i];
        const item = lexToJson(inputItem);
        if (item !== inputItem) {
            copy ?? (copy = Array.from(input));
            copy[i] = item;
        }
    }
    return copy ?? input;
}
function encodeLexMap(input) {
    // Lazily copy value
    let copy = undefined;
    for (const [key, lexValue] of Object.entries(input)) {
        // Prevent prototype pollution
        if (key === '__proto__') {
            throw new TypeError('Invalid key: __proto__');
        }
        // Ignore (strip) undefined values
        if (lexValue === undefined) {
            copy ?? (copy = { ...input });
            delete copy[key];
            continue;
        }
        const jsonValue = lexToJson(lexValue);
        if (jsonValue !== lexValue) {
            copy ?? (copy = { ...input });
            copy[key] = jsonValue;
        }
    }
    return copy ?? input;
}
function parseSpecialJsonObject(input, options) {
    // Hot path: use hints to avoid parsing when possible
    if (input.$link !== undefined) {
        const cid = (0, link_js_1.parseLexLink)(input);
        if (cid)
            return cid;
        if (options.strict)
            throw new TypeError(`Invalid $link object`);
    }
    else if (input.$bytes !== undefined) {
        const bytes = (0, bytes_js_1.parseLexBytes)(input);
        if (bytes)
            return bytes;
        if (options.strict)
            throw new TypeError(`Invalid $bytes object`);
    }
    else if (input.$type !== undefined) {
        // @NOTE Since blobs are "just" regular lex objects with a special shape,
        // and because an object that does not conform to the blob shape would still
        // result in undefined being returned, we only attempt to parse blobs when
        // the strict option is enabled.
        if (options.strict) {
            if (input.$type === 'blob') {
                const blob = (0, blob_js_1.parseBlobRef)(input, options);
                if (blob)
                    return blob;
                throw new TypeError(`Invalid blob object`);
            }
            else if (typeof input.$type !== 'string') {
                throw new TypeError(`Invalid $type property (${typeof input.$type})`);
            }
            else if (input.$type.length === 0) {
                throw new TypeError(`Empty $type property`);
            }
        }
    }
    // @NOTE We ignore legacy blob representation here. They can be handled at the
    // application level if needed.
    return undefined;
}
//# sourceMappingURL=lex-json.js.map