"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PhraseyBuildablePipeline = exports.PhraseyUtils = exports.PhraseySafeResolvePackage = exports.PhraseyTreeLike = exports.PhraseyIdentifierRegex = exports.PhraseyVersion = void 0;
const errors_1 = require("./errors");
// eslint-disable-next-line @typescript-eslint/no-var-requires
const { version } = require("../package.json");
exports.PhraseyVersion = version;
exports.PhraseyIdentifierRegex = /^[A-z][A-z0-9_]*$/;
exports.PhraseyTreeLike = {
    LR: "─",
    TBR: "├",
    TR: "└",
    build: (data, { map, prefix = "", symbolPostMap } = {}) => {
        const lines = [];
        const dataArray = Array.isArray(data) ? data : [data];
        dataArray.forEach((x, i) => {
            let prefixSymbol = i === dataArray.length - 1
                ? exports.PhraseyTreeLike.TR
                : exports.PhraseyTreeLike.TBR;
            prefixSymbol += exports.PhraseyTreeLike.LR;
            if (symbolPostMap) {
                prefixSymbol = symbolPostMap(prefixSymbol);
            }
            const text = exports.PhraseyTreeLike.spacify(`${map?.(x) ?? x}`);
            lines.push(`${prefix}${prefixSymbol} ${text}`);
        });
        return lines.join("\n");
    },
    tab: (count) => "   ".repeat(count),
    spacify: (text) => {
        const lines = text.split("\n").map((x, i) => {
            if (i === 0)
                return x;
            return `${exports.PhraseyTreeLike.tab(1)}${x}`;
        });
        return lines.join("\n");
    },
};
const PhraseySafeResolvePackage = (packageName) => {
    try {
        return require(packageName);
    }
    catch (error) {
        throw new errors_1.PhraseyError(`Cannot import format package named "${packageName}". Did you install it?`);
    }
};
exports.PhraseySafeResolvePackage = PhraseySafeResolvePackage;
class PhraseyUtils {
    static parseStringArrayNullable(data) {
        if (typeof data === "string")
            return [data];
        if (Array.isArray(data))
            return data;
        return [];
    }
    static equals(a, b) {
        const aType = typeof a;
        const bType = typeof b;
        if (aType !== bType) {
            return false;
        }
        if (Array.isArray(a)) {
            return this._arrayEquals(a, b);
        }
        if (aType === "object") {
            return this._objectEquals(a, b);
        }
        return a === b;
    }
    static _arrayEquals(a, b) {
        if (a.length !== b.length) {
            return false;
        }
        return a.every((x, i) => PhraseyUtils.equals(b[i], x));
    }
    static _objectEquals(a, b) {
        const aKeys = Object.keys(a);
        const bKeys = Object.keys(b);
        if (aKeys.length !== bKeys.length) {
            return false;
        }
        return aKeys.every((x) => this.equals(b[x], a[x]));
    }
}
exports.PhraseyUtils = PhraseyUtils;
class PhraseyBuildablePipeline {
    tasks = [];
    add(task) {
        this.tasks.push(task);
    }
    build() {
        return this;
    }
    buildWithOutput(output) {
        return {
            execute: async () => {
                const result = await this.execute();
                if (!result.success) {
                    return result;
                }
                return PhraseyBuildablePipeline.executeTask(output);
            },
        };
    }
    async execute() {
        for (const x of this.tasks) {
            const result = await PhraseyBuildablePipeline.executeTask(x);
            if (!result.success) {
                return result;
            }
        }
        return { success: true, data: true };
    }
    static async executeTask(task) {
        try {
            const result = await task();
            return result;
        }
        catch (error) {
            return {
                success: false,
                error: new errors_1.PhraseyError("Unexpected pipeline task error", {
                    cause: error,
                }),
            };
        }
    }
}
exports.PhraseyBuildablePipeline = PhraseyBuildablePipeline;
