"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitInteractiveCommand = void 0;
const commander_1 = require("commander");
const enquirer_1 = __importDefault(require("enquirer"));
const fs_extra_1 = require("fs-extra");
const path_1 = __importDefault(require("path"));
const __1 = require("../..");
const log_1 = require("../utils/log");
exports.InitInteractiveCommand = new commander_1.Command()
    .name("init-interactive")
    .description("Initialize a new project. (deprecated)")
    .option("--config-file <value>", "Path to config file.")
    .option("--config-format <value>", "Format of config file.")
    .option("--input-files <value>", "Path to input files.")
    .option("--input-format <value>", "Format of input files.")
    .option("--schema-file <value>", "Path to schema file.")
    .option("--schema-format <value>", "Format of schema file.")
    .option("--output-dir <value>", "Path of output folder.")
    .option("--output-format <value>", "Format of output file.")
    .option("--output-string-format <value>", "Format of output strings.")
    .option("--y, --yes", "Yes to all.")
    .action(async (options) => {
    log_1.log.write(log_1.pico.bold(`Thank you for choosing ${log_1.pico.cyan("Phrasey")}!`));
    log_1.log.write("Please answer the below question to initialize the project.");
    log_1.log.ln();
    options.configFile ??= await inquire({
        type: "input",
        message: "Path to config file (eg. ./phrasey.json)",
        validate: inquirerNonEmptyStringValidate("Please enter a valid path or glob"),
    });
    options.configFormat ??= await inquire({
        type: "input",
        message: "Format of config file (eg. json)",
        validate: inquirerNonEmptyStringValidate("Please enter a valid format"),
    });
    const configFormatCheck = isContentFormatInstalled(options.configFormat);
    if (!configFormatCheck.isInstalled) {
        log_1.log.ln();
        logUnableToResolveFormat(configFormatCheck);
        log_1.log.ln();
        process.exit(1);
    }
    options.inputFiles ??= await inquire({
        type: "input",
        message: "Path to input files (supports glob) (eg. ./i18n/**.json)",
        validate: inquirerNonEmptyStringValidate("Please enter a valid path or glob"),
    });
    options.inputFormat ??= await inquire({
        type: "input",
        message: "Format of input files (eg. json)",
        validate: inquirerNonEmptyStringValidate("Please enter a valid format"),
    });
    const inputFormatCheck = isContentFormatInstalled(options.inputFormat);
    if (!inputFormatCheck.isInstalled) {
        log_1.log.ln();
        logUnableToResolveFormat(inputFormatCheck);
        log_1.log.ln();
        process.exit(1);
    }
    options.schemaFile ??= await inquire({
        type: "input",
        message: "Path to schema file (eg. ./phrasey-schema.json)",
        validate: inquirerNonEmptyStringValidate("Please enter a valid path"),
    });
    options.schemaFormat ??= await inquire({
        type: "input",
        message: "Format of schema file (eg. json)",
        validate: inquirerNonEmptyStringValidate("Please enter a valid format"),
    });
    const schemaFormatCheck = isContentFormatInstalled(options.schemaFormat);
    if (!schemaFormatCheck.isInstalled) {
        log_1.log.ln();
        logUnableToResolveFormat(schemaFormatCheck);
        log_1.log.ln();
        process.exit(1);
    }
    options.outputDir ??= await inquire({
        type: "input",
        message: "Path to output directory (eg. ./dist-i18n)",
        validate: inquirerNonEmptyStringValidate("Please enter a valid path"),
    });
    options.outputFormat ??= await inquire({
        type: "input",
        message: "Format of output files (eg. json)",
        validate: inquirerNonEmptyStringValidate("Please enter a valid format"),
    });
    const outputFormatCheck = isContentFormatInstalled(options.outputFormat);
    if (!outputFormatCheck.isInstalled) {
        log_1.log.ln();
        logUnableToResolveFormat(outputFormatCheck);
        log_1.log.ln();
        process.exit(1);
    }
    options.outputStringFormat ??= await inquire({
        type: "input",
        message: "Format of translation strings in output files (eg. format-string)",
        validate: inquirerNonEmptyStringValidate("Please enter a valid format"),
    });
    const outputStringFormatCheck = isTranslationStringFormatInstalled(options.outputStringFormat);
    if (!outputStringFormatCheck.isInstalled) {
        log_1.log.ln();
        logUnableToResolveFormat(outputStringFormatCheck);
        log_1.log.ln();
        process.exit(1);
    }
    options.yes ??= await inquire({
        type: "confirm",
        message: "Proceed?",
        initial: false,
    });
    if (!options.yes) {
        log_1.log.ln();
        log_1.log.warn("Exiting...");
        log_1.log.ln();
        process.exit(1);
    }
    log_1.log.ln();
    const { configFile, configFormat, inputFiles, inputFormat, schemaFile, schemaFormat, outputDir, outputFormat, outputStringFormat, } = options;
    const config = {
        input: {
            files: [inputFiles],
            format: inputFormat,
            fallback: [],
        },
        schema: {
            file: schemaFile,
            format: schemaFormat,
        },
        output: {
            dir: outputDir,
            format: outputFormat,
            stringFormat: outputStringFormat,
        },
        hooks: {
            files: [],
        },
    };
    const configFormatter = (0, __1.PhraseySafeRun)(() => __1.PhraseyContentFormats.resolve(configFormat));
    if (configFormatter.success) {
        const configFilePath = path_1.default.resolve(process.cwd(), configFile);
        await (0, fs_extra_1.writeFile)(configFilePath, configFormatter.data.serialize(config));
        log_1.log.success(`Generated config at "${configFilePath}".`);
    }
    else {
        log_1.log.error(`Unable to use specified config format "${configFormat}" due to errors.`);
        log_1.log.logErrors(configFormatter.error);
        log_1.log.ln();
        log_1.log.info(`You could try manually creating "${configFile}" using the below generated config.`);
        log_1.log.write(JSON.stringify(config, null, 4));
        log_1.log.ln();
    }
    const schema = {
        keys: [
            {
                name: "HelloX",
                description: "Say hello to a user!",
                parameters: ["user"],
            },
        ],
    };
    const schemaFormatter = (0, __1.PhraseySafeRun)(() => __1.PhraseyContentFormats.resolve(schemaFormat));
    if (schemaFormatter.success) {
        const schemaFilePath = path_1.default.resolve(process.cwd(), schemaFile);
        await (0, fs_extra_1.writeFile)(schemaFilePath, schemaFormatter.data.serialize(schema));
        log_1.log.success(`Generated schema at "${schemaFilePath}".`);
    }
    else {
        log_1.log.error(`Unable to use specified schema format "${configFormat}" due to errors.`);
        log_1.log.logErrors(schemaFormatter.error);
        log_1.log.ln();
        log_1.log.info(`You could try manually creating "${schemaFile}" using the below generated schema.`);
        log_1.log.write(JSON.stringify(schema, null, 4));
        log_1.log.ln();
    }
    const demoTranslation = {
        locale: "en",
        keys: {
            HelloX: "Hello {user}!",
        },
    };
    const inputFormatter = (0, __1.PhraseySafeRun)(() => __1.PhraseyContentFormats.resolve(inputFormat));
    if (inputFormatter.success) {
        const serializedDemoTranslation = inputFormatter.data.serialize(demoTranslation);
        log_1.log.info("Get started by creating a translation file that matches the generated content!");
        log_1.log.info("Example of translation file:");
        log_1.log.write(serializedDemoTranslation.trim());
        log_1.log.ln();
    }
    else {
        log_1.log.error(`Unable to use specified input translation format "${inputFormat}" due to errors.`);
        log_1.log.logErrors(inputFormatter.error);
        log_1.log.ln();
        log_1.log.info("You could try manually creating a translation file using the below generated content.");
        log_1.log.write(JSON.stringify(demoTranslation, null, 4));
        log_1.log.ln();
    }
});
function logUnableToResolveFormat(result) {
    log_1.log.error(`Unable to resolve format "${result.format}".`);
    log_1.log.logErrors(result.error);
}
function isContentFormatInstalled(format) {
    const packageName = __1.PhraseyContentFormats.defaultPackages[format] ?? format;
    try {
        __1.PhraseyContentFormats.resolve(packageName);
        return { format, isInstalled: true, packageName };
    }
    catch (error) {
        return { format, isInstalled: false, packageName, error };
    }
}
function isTranslationStringFormatInstalled(format) {
    const packageName = format;
    try {
        return {
            format,
            isInstalled: !!__1.PhraseyTranslationStringFormats.defaultFormats[format] ||
                !!__1.PhraseyTranslationStringFormats.resolve(packageName),
            packageName,
        };
    }
    catch (error) {
        return { format, isInstalled: false, packageName, error };
    }
}
async function inquire(options) {
    const { question } = await enquirer_1.default.prompt({
        name: "question",
        ...options,
    });
    return question;
}
function inquirerNonEmptyStringValidate(error) {
    return (input) => {
        if (typeof input === "string" && input.length > 0) {
            return true;
        }
        return error;
    };
}
