"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PhraseyWatcher = void 0;
const chokidar_1 = __importDefault(require("chokidar"));
const builder_1 = require("./builder");
const errors_1 = require("./errors");
const phrasey_1 = require("./phrasey");
const state_1 = require("./state");
const translations_1 = require("./translations");
const utils_1 = require("./utils");
class PhraseyWatcher {
    options;
    listeners = [];
    phrasey;
    state;
    configWatcher;
    localesWatcher;
    schemaWatcher;
    translationsWatcher;
    constructor(options) {
        this.options = options;
        this.phrasey = new phrasey_1.Phrasey(this.options.phrasey);
        this.state = new state_1.PhraseyState(this.phrasey);
    }
    async initialize() {
        await this.updateConfig();
        const configPath = this.phrasey.path(this.options.builder.config.file);
        this.configWatcher = PhraseyWatcher.createWatcher(configPath, () => this.onConfigChange());
    }
    async updateConfig() {
        await this.localesWatcher?.close();
        delete this.localesWatcher;
        await this.schemaWatcher?.close();
        delete this.schemaWatcher;
        await this.translationsWatcher?.close();
        delete this.translationsWatcher;
        const state = new state_1.PhraseyState(this.phrasey);
        const prevState = this.state;
        this.state = state;
        const builder = new builder_1.PhraseyBuilder(this.phrasey, state, this.options.builder);
        const pipeline = builder.constructPipeline({
            loadConfig: true,
            loadHooks: true,
            loadLocales: false,
            loadSchema: false,
            loadTranslations: false,
            ensureTranslations: false,
            buildTranslations: false,
        });
        const result = await pipeline.execute();
        if (!result.success) {
            await this.onError(new errors_1.PhraseyError("Updating config failed", {
                cause: result.error,
            }));
            return;
        }
        this.log.success("Synced config state.");
        const prevConfig = prevState.maybeGetConfig();
        const config = state.getConfig();
        const isLocalesUnchanged = prevState.hasLocales() &&
            utils_1.PhraseyUtils.equals(prevConfig?.z.locales, config.z.locales);
        const isSchemaUnchanged = prevState.hasSchema() &&
            utils_1.PhraseyUtils.equals(prevConfig?.z.schema, config.z.schema);
        const isTranslationsUnchanged = isLocalesUnchanged &&
            isSchemaUnchanged &&
            prevState.hasTranslations() &&
            utils_1.PhraseyUtils.equals(prevConfig?.z.input, config.z.input) &&
            utils_1.PhraseyUtils.equals(prevConfig?.z.output, config.z.output);
        if (isLocalesUnchanged) {
            state.setLocales(prevState.getLocales());
        }
        else {
            await this.onLocalesChange();
        }
        if (isSchemaUnchanged) {
            state.setSchema(prevState.getSchema());
        }
        else {
            await this.onSchemaChange();
        }
        if (isTranslationsUnchanged) {
            state.setTranslations(prevState.getTranslations());
        }
        else {
            await this.onTranslationsChange();
        }
        if (config.z.locales) {
            const localesPath = this.phrasey.path(config.z.locales.file);
            this.localesWatcher = PhraseyWatcher.createWatcher(localesPath, () => this.onLocalesChange());
        }
        const schemaPath = this.phrasey.path(config.z.schema.file);
        this.schemaWatcher = PhraseyWatcher.createWatcher(schemaPath, () => this.onSchemaChange());
        const translationPaths = Array.isArray(config.z.input.files)
            ? config.z.input.files.map((x) => this.phrasey.path(x))
            : this.phrasey.path(config.z.input.files);
        this.translationsWatcher = PhraseyWatcher.createWatcher(translationPaths, (_, path) => this.onTranslationChange(path));
    }
    async updateLocales() {
        const builder = new builder_1.PhraseyBuilder(this.phrasey, this.state, this.options.builder);
        const pipeline = builder.constructPipeline({
            loadConfig: false,
            loadHooks: false,
            loadLocales: true,
            loadSchema: false,
            loadTranslations: false,
            ensureTranslations: false,
            buildTranslations: false,
        });
        const result = await pipeline.execute();
        if (!result.success) {
            await this.onError(new errors_1.PhraseyError("Updating locales failed", {
                cause: result.error,
            }));
            return;
        }
        this.log.success("Synced locales state.");
    }
    async updateSchema() {
        const builder = new builder_1.PhraseyBuilder(this.phrasey, this.state, this.options.builder);
        const pipeline = builder.constructPipeline({
            loadConfig: false,
            loadHooks: false,
            loadLocales: false,
            loadSchema: true,
            loadTranslations: false,
            ensureTranslations: false,
            buildTranslations: false,
        });
        const result = await pipeline.execute();
        if (!result.success) {
            await this.onError(new errors_1.PhraseyError("Updating schema failed", {
                cause: result.error,
            }));
            return;
        }
        this.log.success("Synced schema state.");
    }
    async updateTranslations() {
        this.state.setTranslations(new translations_1.PhraseyTranslations(this.state.getSchema()));
        if (this.options.watcher.buildAllTranslations) {
            const builder = new builder_1.PhraseyBuilder(this.phrasey, this.state, this.options.builder);
            const pipeline = builder.constructPipeline({
                loadConfig: false,
                loadHooks: false,
                loadLocales: false,
                loadSchema: false,
                loadTranslations: true,
                ensureTranslations: true,
                buildTranslations: true,
            });
            const result = await pipeline.execute();
            if (!result.success) {
                await this.onError(new errors_1.PhraseyError("Updating translations failed", {
                    cause: result.error,
                }));
                return;
            }
            this.log.success("Processed all translations.");
        }
        this.log.success("Synced translations state.");
    }
    async updateTranslation(path) {
        const rebuildPaths = new Set();
        rebuildPaths.add(path);
        const translations = this.state.getTranslations();
        for (const x of translations.values()) {
            if (x.path === path)
                continue;
            if (x.fallback.includes(path)) {
                rebuildPaths.add(x.path);
            }
        }
        const builder = new builder_1.PhraseyBuilder(this.phrasey, this.state, {
            ...this.options.builder,
            translations: {
                ...this.options.builder?.translations,
                skip: (x) => !rebuildPaths.has(x),
            },
        });
        const pipeline = builder.constructPipeline({
            loadConfig: false,
            loadHooks: false,
            loadLocales: false,
            loadSchema: false,
            loadTranslations: true,
            ensureTranslations: true,
            buildTranslations: true,
        });
        const result = await pipeline.execute();
        if (!result.success) {
            await this.onError(new errors_1.PhraseyError("Updating translation failed", {
                cause: result.error,
            }));
            return;
        }
        this.log.success(`Synced "${path}" translation state.`);
    }
    async onConfigChange() {
        this.log.info("Triggering config change...");
        await this.updateConfig();
        await this.dispatch(async (x) => x.onConfigChange?.());
    }
    async onLocalesChange() {
        this.log.info("Triggering locales change...");
        await this.updateLocales();
        await this.dispatch(async (x) => x.onLocalesChange?.());
    }
    async onSchemaChange() {
        this.log.info("Triggering schema change...");
        await this.updateSchema();
        await this.dispatch(async (x) => x.onSchemaChange?.());
    }
    async onTranslationsChange() {
        this.log.info("Triggering translations change...");
        await this.updateTranslations();
        await this.dispatch(async (x) => x.onTranslationsChange?.());
    }
    async onTranslationChange(path) {
        this.log.info(`Triggering "${path}" translation change...`);
        await this.updateTranslation(path);
        await this.dispatch(async (x) => x.onTranslationChange?.(path));
    }
    async onError(error) {
        this.log.error("Watcher encountered an error.");
        this.log.logErrors(error);
        await this.dispatch(async (x) => x.onError?.(error));
    }
    async destroy() {
        await this.configWatcher?.close();
        delete this.configWatcher;
        await this.localesWatcher?.close();
        delete this.localesWatcher;
        await this.schemaWatcher?.close();
        delete this.schemaWatcher;
        await this.translationsWatcher?.close();
        delete this.translationsWatcher;
    }
    listen(listener) {
        this.listeners.push(listener);
        return () => this.unlisten(listener);
    }
    unlisten(listener) {
        this.listeners = this.listeners.filter((x) => x !== listener);
    }
    async dispatch(apply) {
        for (const x of this.listeners) {
            try {
                await apply(x);
            }
            catch (error) {
                await this.onError(new errors_1.PhraseyError("Event dispatching failed", {
                    cause: error,
                }));
            }
        }
    }
    get log() {
        return this.phrasey.log;
    }
    static async create(options) {
        const watcher = new PhraseyWatcher(options);
        await watcher.initialize();
        return watcher;
    }
    static createWatcher(path, listener) {
        const watcher = chokidar_1.default.watch(path, {
            ignoreInitial: true,
        });
        watcher.on("all", listener);
        return watcher;
    }
}
exports.PhraseyWatcher = PhraseyWatcher;
