"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PhraseyTranslations = void 0;
const errors_1 = require("./errors");
const translation_1 = require("./translation");
class PhraseyTranslations {
    schema;
    translations = new Map();
    pathCodes = new Map();
    constructor(schema) {
        this.schema = schema;
    }
    async load(path, formatter, locales, globalFallback) {
        const translation = await translation_1.PhraseyTranslation.create(path, this.schema, formatter, locales, globalFallback);
        if (!translation.success)
            return translation;
        const localeCode = translation.data.locale.code;
        this.translations.set(localeCode, translation.data);
        this.pathCodes.set(PhraseyTranslations.normalizePath(path), localeCode);
        return { success: true, data: localeCode };
    }
    ensure(translation) {
        if (this.schema.keysCount() === translation.keysCount()) {
            return { success: true, data: true };
        }
        const fallback = [];
        for (const x of translation.fallback) {
            const locale = this.pathCodes.get(PhraseyTranslations.normalizePath(x));
            if (!locale) {
                return {
                    success: false,
                    error: new errors_1.PhraseyError(`Invalid translation file path "${x}"`),
                };
            }
            const resolved = this.translations.get(locale);
            if (!resolved) {
                return {
                    success: false,
                    error: new errors_1.PhraseyError(`Unknown translation "${locale}"`),
                };
            }
            fallback.push(resolved);
        }
        for (const x of this.schema.keys.values()) {
            if (translation.hasKey(x.name))
                continue;
            const fallbackValue = PhraseyTranslations.resolveFallbackKey(fallback, x.name);
            switch (fallbackValue?.state) {
                case "set":
                case "fallback":
                    translation.setKey(x.name, {
                        state: "fallback",
                        parts: fallbackValue.parts,
                    });
                    break;
                default:
                    translation.setKey(x.name, {
                        state: "unset",
                    });
                    break;
            }
        }
        return { success: true, data: true };
    }
    values() {
        return this.translations.values();
    }
    static resolveFallbackKey(translations, key) {
        for (const x of translations) {
            const value = x.getKey(key);
            if (value)
                return value;
        }
        return;
    }
    static normalizePath(path) {
        return path.replace(/\\/g, "/");
    }
}
exports.PhraseyTranslations = PhraseyTranslations;
