"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PhraseyTranslationStringFormats = void 0;
const errors_1 = require("./errors");
const utils_1 = require("./utils");
class PhraseyTranslationStringFormats {
    static defaultFormats = {
        parts: this.construct({
            format: (parts) => parts,
        }),
        /**
         *  0 - String, 1 - Parameter
         */
        "compact-parts": this.construct({
            format: (parts) => {
                return parts.map((x) => {
                    switch (x.type) {
                        case "string":
                            return [0, x.value];
                        case "parameter":
                            return [1, x.value];
                    }
                });
            },
        }),
        "format-string": this.construct({
            format: (parts, schema) => {
                const parameters = schema.parameters ?? [];
                let out = "";
                for (const x of parts) {
                    switch (x.type) {
                        case "parameter":
                            out += `%${parameters.indexOf(x.value)}$s`;
                            break;
                        case "string":
                            out += this.replaceCharacter(x.value, "%", "%%");
                            break;
                    }
                }
                return out;
            },
        }),
        "java-format-string": this.construct({
            format: (parts, schema) => {
                const parameters = schema.parameters ?? [];
                let out = "";
                for (const x of parts) {
                    switch (x.type) {
                        case "parameter":
                            out += `%${parameters.indexOf(x.value) + 1}$s`;
                            break;
                        case "string":
                            out += this.replaceCharacter(x.value, "%", "%%");
                            break;
                    }
                }
                return out;
            },
        }),
        "python-format-string": this.construct({
            format: (parts) => {
                let out = "";
                parts.forEach((x) => {
                    switch (x.type) {
                        case "parameter":
                            out += `{${x.value}}`;
                            break;
                        case "string":
                            // eslint-disable-next-line no-case-declarations
                            let escaped = this.replaceCharacters(x.value, {
                                "{": "{{",
                                "}": "}}",
                            });
                            escaped = this.replaceCharacter(escaped, "}", "}}");
                            out += escaped;
                            break;
                    }
                });
                return out;
            },
        }),
        "python-positional-format-string": this.construct({
            format: (parts, schema) => {
                const parameters = schema.parameters ?? [];
                let out = "";
                parts.forEach((x) => {
                    switch (x.type) {
                        case "parameter":
                            out += `{${parameters.indexOf(x.value)}}`;
                            break;
                        case "string":
                            // eslint-disable-next-line no-case-declarations
                            let escaped = this.replaceCharacters(x.value, {
                                "{": "{{",
                                "}": "}}",
                            });
                            escaped = this.replaceCharacter(escaped, "}", "}}");
                            out += escaped;
                            break;
                    }
                });
                return out;
            },
        }),
    };
    static resolve(name) {
        const defaultFormat = this.defaultFormats[name];
        if (defaultFormat) {
            return defaultFormat;
        }
        const pkg = (0, utils_1.PhraseySafeResolvePackage)(name);
        if (typeof pkg?.stringFormatter !== "object") {
            throw new errors_1.PhraseyError(`Missing implementation of "stringFormatter" in package "${name}"`);
        }
        if (typeof pkg.stringFormatter.format !== "function") {
            throw new errors_1.PhraseyError(`Missing implementation of "stringFormatter.format" in package "${name}"`);
        }
        return pkg.stringFormatter;
    }
    static replaceCharacter(value, from, to) {
        let out = "";
        for (let i = 0; i < value.length; i++) {
            const char = value[i];
            if (char === from) {
                out += to;
            }
            out += char;
        }
        return out;
    }
    static replaceCharacters(value, replacers) {
        let out = "";
        for (let i = 0; i < value.length; i++) {
            const char = value[i];
            out += replacers[char] ?? char;
        }
        return out;
    }
    static construct(formatter) {
        return formatter;
    }
}
exports.PhraseyTranslationStringFormats = PhraseyTranslationStringFormats;
