"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PhraseyTranslationStats = exports.PhraseyTranslation = void 0;
const path_1 = __importDefault(require("path"));
const errors_1 = require("./errors");
const transformer_1 = require("./transformer");
const utils_1 = require("./utils");
const z_1 = require("./z");
class PhraseyTranslation {
    path;
    schema;
    locale;
    extras;
    fallback;
    keys = new Map();
    stats = new PhraseyTranslationStats();
    constructor(path, schema, locale, 
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    extras, fallback) {
        this.path = path;
        this.schema = schema;
        this.locale = locale;
        this.extras = extras;
        this.fallback = fallback;
    }
    setKey(key, value) {
        const pValue = this.keys.get(key);
        if (pValue) {
            this.stats.unprocess(key, pValue);
        }
        this.keys.set(key, value);
        this.stats.process(key, value);
    }
    getKey(key) {
        return this.keys.get(key);
    }
    hasKey(key) {
        return this.keys.has(key);
    }
    keysCount() {
        return this.keys.size;
    }
    serialize(stringFormatter) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        const keys = {};
        for (const [k, v] of this.keys.entries()) {
            if (v.state === "set" || v.state === "fallback") {
                const keySchema = this.schema.key(k);
                try {
                    keys[k] = stringFormatter.format(v.parts, keySchema);
                }
                catch (error) {
                    throw new errors_1.PhraseyError("Formatting translation string failed", { cause: error });
                }
            }
        }
        return { locale: this.locale, extras: this.extras, keys };
    }
    json() {
        return {
            locale: this.locale,
            extras: this.extras,
            keys: Object.fromEntries(this.keys.entries()),
        };
    }
    static async create(path, schema, formatter, locales, globalFallback) {
        const unprocessed = await transformer_1.PhraseyTransformer.transform(path, formatter, z_1.PhraseyZTranslation);
        if (!unprocessed.success) {
            return { success: false, error: unprocessed.error };
        }
        const locale = locales.all.find((x) => x.code === unprocessed.data.locale);
        if (!locale) {
            return {
                success: false,
                error: new errors_1.PhraseyError(`Invalid locale "${unprocessed.data.locale}"`),
            };
        }
        const extras = unprocessed.data.extras ?? {};
        const dir = path_1.default.dirname(path);
        const fallback = utils_1.PhraseyUtils.parseStringArrayNullable(unprocessed.data.fallback).map((x) => path_1.default.resolve(dir, x));
        fallback.push(...globalFallback);
        const translation = new PhraseyTranslation(path, schema, locale, extras, fallback);
        for (const x of schema.z.keys) {
            const rawValue = unprocessed.data.keys[x.name];
            if (!rawValue)
                continue;
            const parts = PhraseyTranslation.parseTranslationKeyValue(x, rawValue);
            if (!parts.success)
                return parts;
            translation.setKey(x.name, {
                state: "set",
                parts: parts.data,
            });
        }
        return {
            success: true,
            data: translation,
        };
    }
    static parseTranslationKeyValue(key, content) {
        const parts = [];
        let escaped = false;
        let mode = "string";
        let current = "";
        for (let i = 0; i < content.length; i++) {
            const char = content[i];
            if (char === "{" && !escaped) {
                if (mode === "parameter") {
                    return {
                        success: false,
                        error: new errors_1.PhraseyError(`Unexpected delimiter "{" at ${i}`),
                    };
                }
                if (current.length > 0) {
                    parts.push({
                        type: "string",
                        value: current,
                    });
                }
                current = "";
                mode = "parameter";
            }
            else if (char === "}" && mode === "parameter") {
                if (!(key.parameters?.includes(current) ?? false)) {
                    return {
                        success: false,
                        error: new errors_1.PhraseyError(`Invalid parameter "${current}" at ${i - current.length - 1}`),
                    };
                }
                parts.push({
                    type: "parameter",
                    value: current,
                });
                current = "";
                mode = "string";
            }
            else if (char === "\\") {
                escaped = true;
            }
            else {
                current += char;
            }
        }
        if (current.length > 0) {
            parts.push({
                type: "string",
                value: current,
            });
        }
        return { success: true, data: parts };
    }
}
exports.PhraseyTranslation = PhraseyTranslation;
class PhraseyTranslationStats {
    set = new Set();
    fallback = new Set();
    unset = new Set();
    total = 0;
    process(key, value) {
        switch (value.state) {
            case "set":
                this.set.add(key);
                this.total++;
                break;
            case "fallback":
                this.fallback.add(key);
                this.total++;
                break;
            case "unset":
                this.unset.add(key);
                this.total++;
                break;
        }
    }
    unprocess(key, value) {
        switch (value.state) {
            case "set":
                this.set.delete(key);
                this.total--;
                break;
            case "fallback":
                this.fallback.delete(key);
                this.total--;
                break;
            case "unset":
                this.unset.delete(key);
                this.total--;
                break;
        }
    }
    json() {
        return {
            set: {
                keys: [...this.set],
                count: this.setCount,
                percent: this.setPercent,
            },
            fallback: {
                keys: [...this.fallback],
                count: this.fallbackCount,
                percent: this.fallbackPercent,
            },
            unset: {
                keys: [...this.unset],
                count: this.unsetCount,
                percent: this.unsetPercent,
            },
            total: this.total,
            isBuildable: this.isBuildable,
            isStandaloneBuildable: this.isStandaloneBuildable,
        };
    }
    get isBuildable() {
        return this.setCount + this.fallbackCount === this.total;
    }
    get isStandaloneBuildable() {
        return this.setCount === this.total;
    }
    get setCount() {
        return this.set.size;
    }
    get fallbackCount() {
        return this.fallback.size;
    }
    get unsetCount() {
        return this.unset.size;
    }
    get setPercent() {
        return (this.setCount / this.total) * 100;
    }
    get fallbackPercent() {
        return (this.fallbackCount / this.total) * 100;
    }
    get unsetPercent() {
        return (this.unsetCount / this.total) * 100;
    }
}
exports.PhraseyTranslationStats = PhraseyTranslationStats;
