"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PhraseyBuilder = void 0;
const fast_glob_1 = __importDefault(require("fast-glob"));
const fs_extra_1 = require("fs-extra");
const config_1 = require("./config");
const contentFormats_1 = require("./contentFormats");
const errors_1 = require("./errors");
const hooks_1 = require("./hooks");
const locales_1 = require("./locales");
const phrasey_1 = require("./phrasey");
const result_1 = require("./result");
const schema_1 = require("./schema");
const state_1 = require("./state");
const summary_1 = require("./summary");
const translationStringFormat_1 = require("./translationStringFormat");
const translations_1 = require("./translations");
const utils_1 = require("./utils");
class PhraseyBuilder {
    phrasey;
    state;
    options;
    constructor(phrasey, state, options) {
        this.phrasey = phrasey;
        this.state = state;
        this.options = options;
    }
    async loadConfig() {
        const configPath = this.phrasey.path(this.options.config.file);
        const configResult = await config_1.PhraseyConfig.create(configPath, this.options.config.format);
        if (!configResult.success) {
            return configResult;
        }
        this.state.setConfig(configResult.data);
        return { success: true, data: true };
    }
    async loadHooks() {
        const hooks = new hooks_1.PhraseyHooks(this.phrasey);
        for (const x of this.config.z.hooks?.files ?? []) {
            const hookPath = this.phrasey.path(x.path);
            const hookResult = hooks.addHandlerFile(hookPath, x.options ?? {});
            if (!hookResult.success) {
                return hookResult;
            }
        }
        this.state.setHooks(hooks);
        return { success: true, data: true };
    }
    async loadLocales() {
        const beforeHookResult = await this.hooks.dispatch("beforeLocalesParsing", this.state, {});
        if (!beforeHookResult.success) {
            return beforeHookResult;
        }
        if (this.config.z.locales) {
            const path = this.phrasey.path(this.config.z.locales.file);
            const result = await locales_1.PhraseyLocales.create(path, this.config.z.locales.format);
            if (!result.success) {
                return result;
            }
            this.state.setLocales(result.data);
        }
        else {
            const locales = locales_1.PhraseyLocales.defaultLocales();
            this.state.setLocales(locales);
        }
        const afterHookResult = await this.hooks.dispatch("onLocalesParsed", this.state, {});
        if (!afterHookResult.success) {
            return afterHookResult;
        }
        return { success: true, data: true };
    }
    async loadSchema() {
        const beforeHookResult = await this.hooks.dispatch("beforeSchemaParsing", this.state, {});
        if (!beforeHookResult.success) {
            return beforeHookResult;
        }
        const schemaPath = this.phrasey.path(this.config.z.schema.file);
        const schemaResult = await schema_1.PhraseySchema.create(schemaPath, this.config.z.schema.format);
        if (!schemaResult.success) {
            return schemaResult;
        }
        this.state.setSchema(schemaResult.data);
        const afterHookResult = await this.hooks.dispatch("onSchemaParsed", this.state, {});
        if (!afterHookResult.success) {
            return afterHookResult;
        }
        return { success: true, data: true };
    }
    getInputFallbackFilePaths() {
        const paths = utils_1.PhraseyUtils.parseStringArrayNullable(this.config.z.input.fallback).map((x) => this.phrasey.path(x));
        return paths;
    }
    getInputFilePaths() {
        const stream = fast_glob_1.default.stream(this.config.z.input.files, {
            cwd: this.phrasey.cwd,
            absolute: true,
            onlyFiles: true,
        });
        return stream;
    }
    async loadTranslations() {
        const beforeHookResult = await this.hooks.dispatch("beforeTranslationsParsing", this.state, {});
        if (!beforeHookResult.success) {
            return beforeHookResult;
        }
        const formatter = contentFormats_1.PhraseyContentFormats.resolve(this.config.z.input.format);
        const globalFallback = this.getInputFallbackFilePaths();
        const stream = this.getInputFilePaths();
        this.state.setTranslations(new translations_1.PhraseyTranslations(this.schema));
        const errors = [];
        for await (const x of stream) {
            const path = x.toString();
            if (this.options.translations?.skip?.(path)) {
                continue;
            }
            const result = await this.loadTranslation(path, formatter, globalFallback);
            if (!result.success) {
                errors.push(result.error);
            }
        }
        if (errors.length > 0) {
            return { success: false, error: errors };
        }
        const afterHookResult = await this.hooks.dispatch("onTranslationsParsed", this.state, {});
        if (!afterHookResult.success) {
            return afterHookResult;
        }
        return { success: true, data: true };
    }
    async loadTranslation(path, formatter, globalFallback) {
        const beforeHookResult = await this.hooks.dispatch("beforeTranslationParsing", this.state, { path });
        if (!beforeHookResult.success) {
            return beforeHookResult;
        }
        const result = await this.translations.load(path, formatter, this.locales, globalFallback);
        if (!result.success) {
            return result;
        }
        const afterHookResult = await this.hooks.dispatch("onTranslationParsed", this.state, { locale: result.data });
        if (!afterHookResult.success) {
            return afterHookResult;
        }
        return { success: true, data: true };
    }
    async ensureTranslations() {
        const beforeHookResult = await this.hooks.dispatch("beforeTranslationsEnsuring", this.state, {});
        if (!beforeHookResult.success) {
            return beforeHookResult;
        }
        const errors = [];
        for (const x of this.translations.values()) {
            if (this.options.translations?.skip?.(x.path)) {
                continue;
            }
            const result = await this.ensureTranslation(x);
            if (!result.success) {
                errors.push(result.error);
            }
        }
        if (errors.length > 0) {
            return { success: false, error: errors };
        }
        const afterHookResult = await this.hooks.dispatch("onTranslationsEnsured", this.state, {});
        if (!afterHookResult.success) {
            return afterHookResult;
        }
        return { success: true, data: true };
    }
    async ensureTranslation(translation) {
        const localeCode = translation.locale.code;
        const beforeHookResult = await this.hooks.dispatch("beforeTranslationEnsuring", this.state, { locale: localeCode });
        if (!beforeHookResult.success) {
            return beforeHookResult;
        }
        const result = this.translations.ensure(translation);
        if (!result.success) {
            return result;
        }
        const afterHookResult = await this.hooks.dispatch("onTranslationEnsured", this.state, { locale: localeCode });
        if (!afterHookResult.success) {
            return afterHookResult;
        }
        return { success: true, data: true };
    }
    async buildTranslations() {
        const beforeHookResult = await this.hooks.dispatch("beforeTranslationsBuilding", this.state, {});
        if (!beforeHookResult.success) {
            return beforeHookResult;
        }
        if (!this.config.z.output) {
            return {
                success: false,
                error: new errors_1.PhraseyError('Cannot build when "output" is not specified in configuration file'),
            };
        }
        const formatter = contentFormats_1.PhraseyContentFormats.resolve(this.config.z.output.format);
        const stringFormatter = translationStringFormat_1.PhraseyTranslationStringFormats.resolve(this.config.z.output.stringFormat);
        const errors = [];
        for (const x of this.translations.values()) {
            if (this.options.translations?.skip?.(x.path)) {
                continue;
            }
            const path = this.phrasey.path(this.config.z.output.dir, `${x.locale.code}.${formatter.extension}`);
            const result = await this.buildTranslation(path, x, formatter, stringFormatter);
            if (!result.success) {
                errors.push(result.error);
            }
        }
        if (errors.length > 0) {
            return { success: false, error: errors };
        }
        const afterHookResult = await this.hooks.dispatch("onTranslationsBuildFinished", this.state, {});
        if (!afterHookResult.success) {
            return afterHookResult;
        }
        return { success: true, data: true };
    }
    async buildTranslation(path, translation, formatter, stringFormatter) {
        const localeCode = translation.locale.code;
        const beforeHookResult = await this.hooks.dispatch("beforeTranslationBuilding", this.state, { locale: localeCode });
        if (!beforeHookResult.success) {
            return beforeHookResult;
        }
        if (!translation.stats.isBuildable) {
            return {
                success: false,
                error: new errors_1.PhraseyError(`Translation "${localeCode}" is not buildable`),
            };
        }
        await (0, fs_extra_1.ensureFile)(path);
        const content = (0, result_1.PhraseySafeRun)(() => formatter.serialize(translation.serialize(stringFormatter)));
        if (!content.success) {
            return {
                success: false,
                error: new errors_1.PhraseyError(`Serializing "${localeCode}" failed`, {
                    cause: content.error,
                }),
            };
        }
        await (0, fs_extra_1.writeFile)(path, content.data);
        const afterHookResult = await this.hooks.dispatch("onTranslationBuildFinished", this.state, { locale: localeCode });
        if (!afterHookResult.success) {
            return afterHookResult;
        }
        return { success: true, data: true };
    }
    getSummary() {
        const summary = new summary_1.PhraseySummary({
            keysCount: this.schema.keysCount(),
        });
        for (const x of this.translations.values()) {
            summary.add(x);
        }
        return summary;
    }
    constructBuildablePipeline(options) {
        const pipeline = new utils_1.PhraseyBuildablePipeline();
        if (options.loadConfig) {
            pipeline.add(this.loadConfig.bind(this));
        }
        if (options.loadHooks) {
            pipeline.add(this.loadHooks.bind(this));
        }
        if (options.loadLocales) {
            pipeline.add(this.loadLocales.bind(this));
        }
        if (options.loadSchema) {
            pipeline.add(this.loadSchema.bind(this));
        }
        if (options.loadTranslations) {
            pipeline.add(this.loadTranslations.bind(this));
        }
        if (options.ensureTranslations) {
            pipeline.add(this.ensureTranslations.bind(this));
        }
        if (options.buildTranslations) {
            pipeline.add(this.buildTranslations.bind(this));
        }
        return pipeline;
    }
    constructPipeline(options) {
        const pipeline = this.constructBuildablePipeline(options);
        return pipeline.build();
    }
    constructBuildPipeline() {
        const pipeline = this.constructPipeline({
            loadConfig: true,
            loadHooks: true,
            loadLocales: true,
            loadSchema: true,
            loadTranslations: true,
            ensureTranslations: true,
            buildTranslations: true,
        });
        return pipeline;
    }
    constructSummaryPipeline() {
        const pipeline = this.constructBuildablePipeline({
            loadConfig: true,
            loadHooks: true,
            loadLocales: true,
            loadSchema: true,
            loadTranslations: true,
            ensureTranslations: true,
            buildTranslations: false,
        });
        return pipeline.buildWithOutput(async () => {
            const summary = new summary_1.PhraseySummary({
                keysCount: this.schema.keysCount(),
            });
            for (const x of this.translations.values()) {
                summary.add(x);
            }
            return { success: true, data: summary };
        });
    }
    get config() {
        return this.state.getConfig();
    }
    get hooks() {
        return this.state.getHooks();
    }
    get schema() {
        return this.state.getSchema();
    }
    get locales() {
        return this.state.getLocales();
    }
    get translations() {
        return this.state.getTranslations();
    }
    static create(options) {
        const phrasey = new phrasey_1.Phrasey(options.phrasey);
        const state = new state_1.PhraseyState(phrasey);
        const builder = new PhraseyBuilder(phrasey, state, options.builder);
        return builder;
    }
    static async build(options) {
        const builder = PhraseyBuilder.create(options);
        const pipeline = builder.constructBuildPipeline();
        return pipeline.execute();
    }
    static async summary(options) {
        const builder = PhraseyBuilder.create(options);
        const pipeline = builder.constructSummaryPipeline();
        return pipeline.execute();
    }
}
exports.PhraseyBuilder = PhraseyBuilder;
