import { Config, Environment, FormatterOptions } from "@taplo/core";
/**
 * Options for the format function.
 */
export interface FormatOptions {
    /**
     * Options to pass to the formatter.
     */
    options?: FormatterOptions;
    /**
     * Taplo configuration, this can be parsed
     * from files like `taplo.toml` or provided manually.
     */
    config?: Config;
}
/**
 * Options for TOML Lint.
 */
export interface LintOptions {
    /**
     * Taplo configuration, this can be parsed
     * from `.taplo.toml` or provided manually.
     */
    config?: Config;
}
/**
 * An lint error.
 */
export interface LintError {
    /**
     * A range within the TOML document if any.
     */
    range?: Range;
    /**
     * The error message.
     */
    error: string;
}
/**
 * The object returned from the lint function.
 */
export interface LintResult {
    /**
     * Lint errors, if any.
     *
     * This includes syntax, semantic and schema errors as well.
     */
    errors: Array<LintError>;
}
/**
 * This class allows for usage of the library in a synchronous context
 * after being asynchronously initialized once.
 *
 * It cannot be constructed with `new`, and instead must be
 * created by calling `initialize`.
 *
 * Example usage:
 *
 * ```js
 * import { Taplo } from "taplo";
 *
 * // Somewhere at the start of your app.
 * const taplo = await Taplo.initialize();
 * // ...
 * // The other methods will not return promises.
 * const formatted = taplo.format(tomlDocument);
 * ```
 */
export declare class Taplo {
    private env;
    private static taplo;
    private static initializing;
    private constructor();
    static initialize(env?: Environment): Promise<Taplo>;
    /**
     * Lint a TOML document, this function returns
     * both syntax and semantic (e.g. conflicting keys) errors.
     *
     * If a JSON schema is found in the config, the TOML document will be validated with it
     * only if it is syntactically valid.
     *
     * Example usage:
     *
     * ```js
     * const lintResult = await taplo.lint(tomlDocument, {
     *   config: { schema: { url: "https://example.com/my-schema.json" } },
     * });
     *
     * if (lintResult.errors.length > 0) {
     *   throw new Error("the document is invalid");
     * }
     * ```
     *
     * @param toml TOML document.
     * @param options Optional additional options.
     */
    lint(toml: string, options?: LintOptions): Promise<LintResult>;
    /**
     * Format the given TOML document.
     *
     * @param toml TOML document.
     * @param options Optional format options.
     */
    format(toml: string, options?: FormatOptions): string;
    /**
     * Encode the given JavaScript object to TOML.
     *
     * @throws If the given object cannot be serialized to TOML.
     *
     * @param data JSON compatible JavaScript object or JSON string.
     */
    encode(data: object | string): string;
    /**
     * Decode the given TOML string to a JavaScript object.
     *
     * @throws If data is not valid TOML.
     *
     * @param {string} data TOML string.
     */
    decode<T extends object = any>(data: string): T;
    /**
     * Convert the given TOML string to JSON.
     *
     * @throws If data is not valid TOML.
     *
     * @param data TOML string.
     * @param {boolean} parse Whether to keep the JSON in a string format.
     */
    decode(data: string, parse: false): string;
}
