package snd.komelia

import android.app.Activity
import android.content.Context
import androidx.datastore.core.DataStoreFactory
import androidx.datastore.dataStoreFile
import coil3.ImageLoader
import coil3.PlatformContext
import coil3.SingletonImageLoader
import coil3.disk.DiskCache
import coil3.network.ktor3.KtorNetworkFetcherFactory
import io.github.oshai.kotlinlogging.KotlinLogging
import io.github.snd_r.komelia.AndroidDependencyContainer
import io.github.snd_r.komelia.AppNotifications
import io.github.snd_r.komelia.fonts.fontsDirectory
import io.github.snd_r.komelia.http.RememberMePersistingCookieStore
import io.github.snd_r.komelia.http.komeliaUserAgent
import io.github.snd_r.komelia.image.AndroidPanelDetector
import io.github.snd_r.komelia.image.AndroidReaderImageFactory
import io.github.snd_r.komelia.image.BookImageLoader
import io.github.snd_r.komelia.image.KomeliaPanelDetector
import io.github.snd_r.komelia.image.ReaderImageFactory
import io.github.snd_r.komelia.image.UpsamplingMode
import io.github.snd_r.komelia.image.coil.CoilDecoder
import io.github.snd_r.komelia.image.coil.FileMapper
import io.github.snd_r.komelia.image.coil.KomgaBookMapper
import io.github.snd_r.komelia.image.coil.KomgaBookPageMapper
import io.github.snd_r.komelia.image.coil.KomgaBookPageThumbnailMapper
import io.github.snd_r.komelia.image.coil.KomgaCollectionMapper
import io.github.snd_r.komelia.image.coil.KomgaReadListMapper
import io.github.snd_r.komelia.image.coil.KomgaSeriesMapper
import io.github.snd_r.komelia.image.coil.KomgaSeriesThumbnailMapper
import io.github.snd_r.komelia.image.processing.ColorCorrectionStep
import io.github.snd_r.komelia.image.processing.CropBordersStep
import io.github.snd_r.komelia.image.processing.ImageProcessingPipeline
import io.github.snd_r.komelia.platform.AndroidWindowState
import io.github.snd_r.komelia.settings.AndroidSecretsRepository
import io.github.snd_r.komelia.settings.AppSettingsSerializer
import io.github.snd_r.komelia.settings.CommonSettingsRepository
import io.github.snd_r.komelia.settings.EpubReaderSettingsRepository
import io.github.snd_r.komelia.settings.ImageReaderSettingsRepository
import io.github.snd_r.komelia.settings.KomfSettingsRepository
import io.github.snd_r.komelia.ui.home.HomeScreenFilterRepository
import io.github.snd_r.komelia.ui.home.homeScreenDefaultFilters
import io.github.snd_r.komelia.updates.AndroidAppUpdater
import io.github.snd_r.komelia.updates.AndroidOnnxModelDownloader
import io.github.snd_r.komelia.updates.UpdateClient
import io.ktor.client.HttpClient
import io.ktor.client.engine.okhttp.OkHttp
import io.ktor.client.plugins.UserAgent
import io.ktor.client.plugins.cookies.HttpCookies
import io.ktor.client.plugins.defaultRequest
import io.ktor.http.Url
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.filterIsInstance
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.io.files.Path
import okhttp3.Cache
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import okio.FileSystem
import snd.komelia.db.AppSettings
import snd.komelia.db.EpubReaderSettings
import snd.komelia.db.ImageReaderSettings
import snd.komelia.db.KomeliaDatabase
import snd.komelia.db.KomfSettings
import snd.komelia.db.SettingsStateActor
import snd.komelia.db.color.ExposedBookColorCorrectionRepository
import snd.komelia.db.color.ExposedColorCurvesPresetRepository
import snd.komelia.db.color.ExposedColorLevelsPresetRepository
import snd.komelia.db.fonts.ExposedUserFontsRepository
import snd.komelia.db.homescreen.ExposedHomeScreenFilterRepository
import snd.komelia.db.repository.ActorEpubReaderSettingsRepository
import snd.komelia.db.repository.ActorHomeScreenFilterRepository
import snd.komelia.db.repository.ActorKomfSettingsRepository
import snd.komelia.db.repository.ActorReaderSettingsRepository
import snd.komelia.db.repository.ActorSettingsRepository
import snd.komelia.db.settings.ExposedEpubReaderSettingsRepository
import snd.komelia.db.settings.ExposedImageReaderSettingsRepository
import snd.komelia.db.settings.ExposedKomfSettingsRepository
import snd.komelia.db.settings.ExposedSettingsRepository
import snd.komelia.image.ImageDecoder
import snd.komelia.image.VipsImageDecoder
import snd.komelia.image.VipsSharedLibrariesLoader
import snd.komelia.onnxruntime.JvmOnnxRuntime
import snd.komelia.onnxruntime.JvmOnnxRuntimeRfDetr
import snd.komelia.onnxruntime.OnnxRuntimeExecutionProvider
import snd.komelia.onnxruntime.OnnxRuntimeSharedLibraries
import snd.komf.client.KomfClientFactory
import snd.komga.client.KomgaClientFactory
import java.util.concurrent.TimeUnit
import kotlin.io.path.createDirectories
import kotlin.time.measureTime

private val logger = KotlinLogging.logger {}

suspend fun initDependencies(
    initScope: CoroutineScope,
    context: Context,
    mainActivity: StateFlow<Activity?>,
): AndroidDependencyContainer {
    measureTime {
        try {
            VipsSharedLibrariesLoader.load()
        } catch (e: UnsatisfiedLinkError) {
            logger.error(e) { "Couldn't load vips shared libraries. reader image loading will not work" }
        }
    }.also { logger.info { "completed vips libraries load in $it" } }

    try {
        OnnxRuntimeSharedLibraries.load()
    } catch (e: UnsatisfiedLinkError) {
        logger.error(e) { "Failed to load onnxruntime " }
    }

    fontsDirectory = Path(context.filesDir.resolve("fonts").absolutePath)

    val datastore = DataStoreFactory.create(
        serializer = AppSettingsSerializer,
        produceFile = { context.dataStoreFile("settings.pb") },
        corruptionHandler = null,
    )
    val secretsRepository = AndroidSecretsRepository(datastore)

    val database = KomeliaDatabase(context.filesDir.resolve("komelia.sqlite").absolutePath)
    val settingsRepository = createCommonSettingsRepository(database)
    val imageReaderSettingsRepository = createImageReaderSettingsRepository(database)
    val epubReaderSettingsRepository = createEpubReaderSettings(database)
    val fontsRepository = ExposedUserFontsRepository(database.database)
    val colorCurvesPresetsRepository = ExposedColorCurvesPresetRepository(database.database)
    val colorLevelsPresetsRepository = ExposedColorLevelsPresetRepository(database.database)
    val bookColorCorrectionRepository = ExposedBookColorCorrectionRepository(database.database)
    val komfSettingsRepository = createKomfSettingsRepository(database)
    val homeScreenFilterRepository = createHomeScreenFilterRepository(database)

    val baseUrl = settingsRepository.getServerUrl().stateIn(initScope)
    val komfUrl = komfSettingsRepository.getKomfUrl().stateIn(initScope)

    val okHttpWithoutCache = createOkHttpClient()
    val okHttpWithCache = okHttpWithoutCache.newBuilder()
        .cache(Cache(directory = context.cacheDir.resolve("okhttp"), maxSize = 50L * 1024L * 1024L))
        .build()
    val ktorWithCache = createKtorClient(okHttpWithCache)
    val ktorWithoutCache = createKtorClient(okHttpWithoutCache)

    val cookiesStorage = RememberMePersistingCookieStore(
        baseUrl.map { Url(it) }.stateIn(initScope),
        secretsRepository
    )
        .also { it.loadRememberMeCookie() }

    val komgaClientFactory = createKomgaClientFactory(
        baseUrl = baseUrl,
        ktorClient = ktorWithCache,
        cookiesStorage = cookiesStorage,
    )

    val colorCorrectionStep = ColorCorrectionStep(bookColorCorrectionRepository)
    val imagePipeline = createImagePipeline(
        cropBorders = imageReaderSettingsRepository.getCropBorders().stateIn(initScope),
        colorCorrectionStep = colorCorrectionStep
    )
    val vipsDecoder = VipsImageDecoder()
    val readerImageFactory = createReaderImageFactory(
        imagePreprocessingPipeline = imagePipeline,
        settings = imageReaderSettingsRepository,
        imageDecoder = vipsDecoder,
        stateFlowScope = initScope
    )

    val readerImageLoader = createReaderImageLoader(
        baseUrl = baseUrl,
        ktorClient = ktorWithoutCache,
        cookiesStorage = cookiesStorage,
        imageFactory = readerImageFactory,
        decoder = vipsDecoder
    )

    val coil = createCoil(
        ktorClient = ktorWithoutCache,
        url = baseUrl,
        cookiesStorage = cookiesStorage,
        decoder = vipsDecoder,
        context = context
    )
    SingletonImageLoader.setSafe { coil }

    val komfClientFactory = KomfClientFactory.Builder()
        .baseUrl { komfUrl.value }
        .ktor(ktorWithCache)
        .build()

    val updateClient = UpdateClient(
        ktor = ktorWithCache,
        ktorWithoutCache = ktorWithoutCache
    )
    val appNotifications = AppNotifications()
    val onnxRuntimeDependencies = createOnnxRuntime(
        context = context,
        updateClient = updateClient,
        appNotifications = appNotifications
    )
    return AndroidDependencyContainer(
        settingsRepository = settingsRepository,
        epubReaderSettingsRepository = epubReaderSettingsRepository,
        imageReaderSettingsRepository = imageReaderSettingsRepository,
        fontsRepository = fontsRepository,
        colorCurvesPresetsRepository = colorCurvesPresetsRepository,
        colorLevelsPresetRepository = colorLevelsPresetsRepository,
        bookColorCorrectionRepository = bookColorCorrectionRepository,
        secretsRepository = secretsRepository,
        komfSettingsRepository = komfSettingsRepository,
        homeScreenFilterRepository = homeScreenFilterRepository,

        appNotifications = appNotifications,
        appUpdater = null,
        komgaClientFactory = komgaClientFactory,
        coilImageLoader = coil,
        platformContext = context,
        bookImageLoader = readerImageLoader,
        komfClientFactory = komfClientFactory,
        windowState = AndroidWindowState(mainActivity),
        imageDecoder = vipsDecoder,
        colorCorrectionStep = colorCorrectionStep,
        readerImageFactory = readerImageFactory,
        onnxRuntime = onnxRuntimeDependencies?.onnxRuntime,
        panelDetector = onnxRuntimeDependencies?.panelDetector,
        onnxModelDownloader = onnxRuntimeDependencies?.onnxModelDownloader
    )
}

private fun createOkHttpClient(): OkHttpClient {
    val loggingInterceptor = HttpLoggingInterceptor { KotlinLogging.logger("http.logging").info { it } }
        .setLevel(HttpLoggingInterceptor.Level.BASIC)
    return OkHttpClient.Builder()
        .connectTimeout(30, TimeUnit.SECONDS)
        .writeTimeout(30, TimeUnit.SECONDS)
        .readTimeout(30, TimeUnit.SECONDS)
        .addInterceptor(loggingInterceptor)
        .build()
}

private fun createKtorClient(
    okHttpClient: OkHttpClient,
): HttpClient {
    return HttpClient(OkHttp) {
        engine { preconfigured = okHttpClient }
        expectSuccess = true

        install(UserAgent) {
            agent = komeliaUserAgent
        }
    }
}

private fun createKomgaClientFactory(
    baseUrl: StateFlow<String>,
    ktorClient: HttpClient,
    cookiesStorage: RememberMePersistingCookieStore,
): KomgaClientFactory {
    return KomgaClientFactory.Builder()
        .ktor(ktorClient)
        .baseUrl { baseUrl.value }
        .cookieStorage(cookiesStorage)
        .build()
}

private fun createReaderImageLoader(
    baseUrl: StateFlow<String>,
    ktorClient: HttpClient,
    cookiesStorage: RememberMePersistingCookieStore,
    decoder: ImageDecoder,
    imageFactory: ReaderImageFactory,
): BookImageLoader {
    val bookClient = KomgaClientFactory.Builder()
        .ktor(ktorClient)
        .baseUrl { baseUrl.value }
        .cookieStorage(cookiesStorage)
        .build()
        .bookClient()
    return BookImageLoader(
        bookClient = bookClient,
        imageDecoder = decoder,
        readerImageFactory = imageFactory,
        diskCache = DiskCache.Builder()
            .directory(FileSystem.SYSTEM_TEMPORARY_DIRECTORY / "komelia_reader_cache")
            .build()
    )
}

private fun createCoil(
    ktorClient: HttpClient,
    url: StateFlow<String>,
    cookiesStorage: RememberMePersistingCookieStore,
    decoder: ImageDecoder,
    context: PlatformContext
): ImageLoader {
    val coilKtorClient = ktorClient.config {
        defaultRequest { url(url.value) }
        install(HttpCookies) { storage = cookiesStorage }
    }
    val diskCache = DiskCache.Builder()
        .directory(FileSystem.SYSTEM_TEMPORARY_DIRECTORY / "coil3_disk_cache")
        .build()
    diskCache.clear()

    return ImageLoader.Builder(context)
        .components {
            add(KomgaBookPageMapper(url))
            add(KomgaBookPageThumbnailMapper(url))
            add(KomgaSeriesMapper(url))
            add(KomgaBookMapper(url))
            add(KomgaCollectionMapper(url))
            add(KomgaReadListMapper(url))
            add(KomgaSeriesThumbnailMapper(url))
            add(FileMapper())
            add(CoilDecoder.Factory(decoder))
            add(KtorNetworkFetcherFactory(httpClient = coilKtorClient))
        }.diskCache(diskCache)
        .build()
}

private fun createImagePipeline(
    cropBorders: StateFlow<Boolean>,
    colorCorrectionStep: ColorCorrectionStep,
): ImageProcessingPipeline {
    val pipeline = ImageProcessingPipeline()
    pipeline.addStep(colorCorrectionStep)
    pipeline.addStep(CropBordersStep(cropBorders))
    return pipeline
}

private suspend fun createCommonSettingsRepository(database: KomeliaDatabase): CommonSettingsRepository {
    val repository = ExposedSettingsRepository(database.database)

    val stateActor = SettingsStateActor(
        settings = repository.get() ?: AppSettings(cardWidth = 150),
        saveSettings = repository::save
    )
    return ActorSettingsRepository(stateActor)
}

private suspend fun createImageReaderSettingsRepository(database: KomeliaDatabase): ImageReaderSettingsRepository {
    val repository = ExposedImageReaderSettingsRepository(database.database)
    val stateActor = SettingsStateActor(
        settings = repository.get() ?: ImageReaderSettings(upsamplingMode = UpsamplingMode.BILINEAR),
        saveSettings = repository::save
    )
    return ActorReaderSettingsRepository(stateActor)
}

private suspend fun createEpubReaderSettings(database: KomeliaDatabase): EpubReaderSettingsRepository {
    val repository = ExposedEpubReaderSettingsRepository(database.database)
    val stateActor = SettingsStateActor(
        settings = repository.get() ?: EpubReaderSettings(),
        saveSettings = repository::save
    )
    return ActorEpubReaderSettingsRepository(stateActor)
}

private suspend fun createKomfSettingsRepository(database: KomeliaDatabase): KomfSettingsRepository {
    val repository = ExposedKomfSettingsRepository(database.database)
    val stateActor = SettingsStateActor(
        settings = repository.get() ?: KomfSettings(),
        saveSettings = repository::save
    )
    return ActorKomfSettingsRepository(stateActor)
}

private suspend fun createHomeScreenFilterRepository(database: KomeliaDatabase): HomeScreenFilterRepository {
    val repository = ExposedHomeScreenFilterRepository(database.database)
    val stateActor = SettingsStateActor(
        settings = repository.getFilters() ?: homeScreenDefaultFilters,
        saveSettings = repository::putFilters
    )
    return ActorHomeScreenFilterRepository(stateActor)
}

private suspend fun createReaderImageFactory(
    imagePreprocessingPipeline: ImageProcessingPipeline,
    settings: ImageReaderSettingsRepository,
    imageDecoder: ImageDecoder,
    stateFlowScope: CoroutineScope,
): AndroidReaderImageFactory {
    return AndroidReaderImageFactory(
        imageDecoder = imageDecoder,
        downSamplingKernel = settings.getDownsamplingKernel().stateIn(stateFlowScope),
        upsamplingMode = settings.getUpsamplingMode().stateIn(stateFlowScope),
        linearLightDownSampling = settings.getLinearLightDownsampling().stateIn(stateFlowScope),
        processingPipeline = imagePreprocessingPipeline,
        stretchImages = settings.getStretchToFit().stateIn(stateFlowScope),
    )
}

private data class OnnxruntimeDependencies(
    val onnxRuntime: JvmOnnxRuntime,
    val panelDetector: KomeliaPanelDetector,
    val onnxModelDownloader: AndroidOnnxModelDownloader,
)

private fun createOnnxRuntime(
    context: Context,
    updateClient: UpdateClient,
    appNotifications: AppNotifications
): OnnxruntimeDependencies? {
    if (!OnnxRuntimeSharedLibraries.isAvailable) return null

    val dataDir = context.dataDir.resolve("onnxruntime").toPath().createDirectories()
    val onnxRuntime = JvmOnnxRuntime.create(dataDir.toString())
    val onnxDir = context.filesDir.resolve("onnx").toPath().createDirectories()
    val rfDetr = JvmOnnxRuntimeRfDetr.create(onnxRuntime)

    val onnxModelDownloader = AndroidOnnxModelDownloader(
        updateClient = updateClient,
        appNotifications = appNotifications,
        dataDir = onnxDir
    )

    val panelDetector = AndroidPanelDetector(
        rfDetr = rfDetr,
        executionProvider = OnnxRuntimeExecutionProvider.CPU,
        deviceId = MutableStateFlow(0),
        updateFlow = onnxModelDownloader.downloadCompletionEvents.filterIsInstance(),
        dataDir = onnxDir,
    ).also { it.initialize() }

    return OnnxruntimeDependencies(
        onnxRuntime = onnxRuntime,
        panelDetector = panelDetector,
        onnxModelDownloader = onnxModelDownloader
    )
}