package geo

import (
	"encoding/json"
	"fmt"
	"strings"

	"github.com/xtls/xray-core/infra/conf"
)

// Read all geo files in config file.
// configPath means where xray config file is.
func ReadGeoFiles(xrayBytes []byte) ([]string, []string) {
	domain, ip := loadXrayConfig(xrayBytes)
	domainCodes := filterAndStrip(domain, "geosite")
	domainFiles := []string{}
	for key := range domainCodes {
		domainFiles = append(domainFiles, key)
	}

	ipCodes := filterAndStrip(ip, "geoip")
	ipFiles := []string{}
	for key := range ipCodes {
		ipFiles = append(ipFiles, key)
	}

	return domainFiles, ipFiles
}

func loadXrayConfig(configBytes []byte) ([]string, []string) {
	domain := []string{}
	ip := []string{}

	var xray *conf.Config
	err := json.Unmarshal(configBytes, &xray)
	if err != nil {
		return domain, ip
	}

	routingDomain, routingIP := filterRouting(xray)
	domain = append(domain, routingDomain...)
	ip = append(ip, routingIP...)

	dnsDomain, dnsIP := filterDns(xray)
	domain = append(domain, dnsDomain...)
	ip = append(ip, dnsIP...)

	return domain, ip
}

func filterRouting(xray *conf.Config) ([]string, []string) {
	domain := []string{}
	ip := []string{}

	routing := xray.RouterConfig
	if routing == nil {
		return domain, ip
	}
	rules := routing.RuleList
	if len(rules) == 0 {
		return domain, ip
	}
	// parse rules
	// we only care about domain and ip
	type RawRule struct {
		Domain *conf.StringList `json:"domain"`
		IP     *conf.StringList `json:"ip"`
	}

	for _, rule := range rules {
		var rawRule RawRule
		err := json.Unmarshal(rule, &rawRule)
		if err != nil {
			continue
		}
		if rawRule.Domain != nil {
			domain = append(domain, *rawRule.Domain...)
		}
		if rawRule.IP != nil {
			ip = append(ip, *rawRule.IP...)
		}
	}
	return domain, ip
}

func filterDns(xray *conf.Config) ([]string, []string) {
	domain := []string{}
	ip := []string{}

	dns := xray.DNSConfig
	if dns == nil {
		return domain, ip
	}
	servers := dns.Servers
	if len(servers) == 0 {
		return domain, ip
	}

	for _, server := range servers {
		if len(server.Domains) > 0 {
			domain = append(domain, server.Domains...)
		}
		if len(server.ExpectIPs) > 0 {
			ip = append(ip, server.ExpectIPs...)
		}
	}
	return domain, ip
}

func filterAndStrip(rules []string, retain string) map[string][]string {
	m := make(map[string][]string)
	retainPrefix := fmt.Sprintf("%s:", retain)
	retainFile := fmt.Sprintf("%s.dat", retain)
	for _, rule := range rules {
		if strings.HasPrefix(rule, retainPrefix) {
			values := strings.SplitN(rule, ":", 2)
			appendMap(m, retainFile, values[1])
		} else if strings.HasPrefix(rule, "ext:") {
			values := strings.SplitN(rule, ":", 3)
			appendMap(m, values[1], values[2])
		}
	}
	return m
}

func appendMap(m map[string][]string, key string, value string) {
	v, ok := m[key]
	if ok {
		v = append(v, value)
	} else {
		v = []string{value}
	}
	m[key] = v
}
