package io.github.potsdam_pnp.initiative_tracker

import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Intent
import android.content.pm.ServiceInfo
import android.os.Build
import androidx.core.app.NotificationCompat
import androidx.lifecycle.LifecycleService
import androidx.lifecycle.lifecycleScope
import io.github.aakira.napier.Napier
import kotlin.time.Duration
import kotlin.time.Duration.Companion.minutes
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.async
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import kotlinx.coroutines.selects.select

class ConnectionService : LifecycleService() {
  private fun channelId(): String {
    val channel =
      NotificationChannel(
          "BACKGROUND_SERVICE_1",
          "background service",
          NotificationManager.IMPORTANCE_LOW,
        )
        .apply { description = "Background service for initiative tracker" }

    (getSystemService(NOTIFICATION_SERVICE) as NotificationManager).apply {
      createNotificationChannel(channel)
    }

    return "BACKGROUND_SERVICE_1"
  }

  override fun onCreate() {
    super.onCreate()

    Napier.i("Service created (1)")

    val notification =
      NotificationCompat.Builder(this, channelId())
        .apply {
          setSmallIcon(R.drawable.ic_notification)
          setContentTitle("Server Running")
          setPriority(NotificationCompat.PRIORITY_LOW)
          setOngoing(true)
          setForegroundServiceBehavior(NotificationCompat.FOREGROUND_SERVICE_IMMEDIATE)
          val actionIntent =
            Intent(this@ConnectionService, ConnectionService::class.java).apply {
              putExtra("stop", true)
            }
          addAction(
            R.drawable.ic_notification,
            "Stop",
            PendingIntent.getService(
              this@ConnectionService,
              1,
              actionIntent,
              PendingIntent.FLAG_IMMUTABLE,
            ),
          )
        }
        .build()

    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
      startForeground(100, notification, ServiceInfo.FOREGROUND_SERVICE_TYPE_REMOTE_MESSAGING)
    } else {
      startForeground(100, notification)
    }

    Napier.i("Service created")

    val app = application as InitiativeTrackerApplication

    server = Server(app.repository)
    app.serverLifecycleManager._serverState.update { server!!.state }
    server!!.toggle(true)
    serverJob = lifecycleScope.launch(Dispatchers.Default) { server!!.runOnce() }
    lifecycleScope.launch {
      delay(1000)
      if (!isShuttingDown) {
        server!!.toggle(true)
      }
    }
    lifecycleScope.launch {
      var currentDelay: Duration? = null

      while (!isShuttingDown) {
        val nextEvent =
          if (currentDelay == null) {
            app.serverLifecycleManager.serverEventChannel.receive()
          } else {
            val delay = currentDelay
            select {
              async { delay(delay) }.onAwait { StopServer }
              app.serverLifecycleManager.serverEventChannel.onReceive { it }
            }
          }
        when (nextEvent) {
          is KeepRunning -> currentDelay = null

          is StopServer ->
            if (!isShuttingDown) {
              app.serverLifecycleManager.startShuttingDown()
              isShuttingDown = true
              server?.toggle(false)

              serverJob?.join()

              stopForeground(STOP_FOREGROUND_REMOVE)
              stopSelf()
              app.serverLifecycleManager.finishedShuttingDown()
            }

          is KillIn -> currentDelay = nextEvent.minutes.minutes
        }
      }
    }
  }

  override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
    if (intent?.getBooleanExtra("stop", false) == true) {
      Napier.i("stop command")
      (application as InitiativeTrackerApplication)
        .serverLifecycleManager
        .serverEventChannel
        .trySend(StopServer)
    } else {
      Napier.i("start command")
      server?.toggle(true)
    }
    return super.onStartCommand(intent, flags, startId)
  }

  override fun onDestroy() {
    super.onDestroy()
  }

  private var isShuttingDown = false
  private var serverJob: Job? = null

  var server: Server? = null
}
