package ly.david.musicsearch.share.feature.database.all

import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.Stable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import com.slack.circuit.foundation.NavEvent
import com.slack.circuit.foundation.onNavEvent
import com.slack.circuit.runtime.CircuitUiEvent
import com.slack.circuit.runtime.CircuitUiState
import com.slack.circuit.runtime.Navigator
import com.slack.circuit.runtime.presenter.Presenter
import ly.david.musicsearch.shared.domain.BrowseMethod
import ly.david.musicsearch.shared.domain.network.MusicBrainzEntityType
import ly.david.musicsearch.ui.common.list.AllEntitiesListPresenter
import ly.david.musicsearch.ui.common.list.AllEntitiesListUiEvent
import ly.david.musicsearch.ui.common.list.AllEntitiesListUiState
import ly.david.musicsearch.ui.common.list.getTotalLocalCount
import ly.david.musicsearch.ui.common.musicbrainz.MusicBrainzLoginPresenter
import ly.david.musicsearch.ui.common.musicbrainz.MusicBrainzLoginUiState
import ly.david.musicsearch.ui.common.screen.AllEntitiesScreen
import ly.david.musicsearch.ui.common.screen.DetailsScreen
import ly.david.musicsearch.ui.common.topappbar.SelectionState
import ly.david.musicsearch.ui.common.topappbar.TopAppBarFilterState
import ly.david.musicsearch.ui.common.topappbar.rememberSelectionState
import ly.david.musicsearch.ui.common.topappbar.rememberTopAppBarFilterState
import ly.david.musicsearch.ui.common.topappbar.toTab

internal class AllLocalEntitiesPresenter(
    private val screen: AllEntitiesScreen,
    private val navigator: Navigator,
    private val allEntitiesListPresenter: AllEntitiesListPresenter,
    private val musicBrainzLoginPresenter: MusicBrainzLoginPresenter,
) : Presenter<AllLocalEntitiesUiState> {

    @Composable
    override fun present(): AllLocalEntitiesUiState {
        var subtitle: String by rememberSaveable { mutableStateOf("") }
        val topAppBarFilterState = rememberTopAppBarFilterState()
        val query = topAppBarFilterState.filterText

        val entitiesListUiState = allEntitiesListPresenter.present()
        val entitiesListEventSink = entitiesListUiState.eventSink

        val selectionState = rememberSelectionState(
            totalCount = entitiesListUiState.getTotalLocalCount(screen.entityType),
        )

        val loginUiState = musicBrainzLoginPresenter.present()

        LaunchedEffect(
            key1 = query,
        ) {
            val browseMethod = BrowseMethod.All
            entitiesListEventSink(
                AllEntitiesListUiEvent.Get(
                    tab = screen.entityType.toTab(),
                    browseMethod = browseMethod,
                    query = query,
                    isRemote = false,
                ),
            )
        }

        fun eventSink(event: AllLocalEntitiesUiEvent) {
            when (event) {
                is AllLocalEntitiesUiEvent.NavigateUp -> {
                    navigator.pop()
                }

                is AllLocalEntitiesUiEvent.ClickItem -> {
                    navigator.onNavEvent(
                        NavEvent.GoTo(
                            DetailsScreen(
                                entityType = event.entityType,
                                id = event.id,
                            ),
                        ),
                    )
                }
            }
        }

        return AllLocalEntitiesUiState(
            subtitle = subtitle,
            entityType = screen.entityType,
            topAppBarFilterState = topAppBarFilterState,
            selectionState = selectionState,
            allEntitiesListUiState = entitiesListUiState,
            musicBrainzLoginUiState = loginUiState,
            eventSink = ::eventSink,
        )
    }
}

@Stable
internal data class AllLocalEntitiesUiState(
    val subtitle: String,
    val entityType: MusicBrainzEntityType,
    val topAppBarFilterState: TopAppBarFilterState = TopAppBarFilterState(),
    val selectionState: SelectionState = SelectionState(),
    val allEntitiesListUiState: AllEntitiesListUiState,
    val musicBrainzLoginUiState: MusicBrainzLoginUiState = MusicBrainzLoginUiState(),
    val eventSink: (AllLocalEntitiesUiEvent) -> Unit,
) : CircuitUiState

internal sealed interface AllLocalEntitiesUiEvent : CircuitUiEvent {
    data object NavigateUp : AllLocalEntitiesUiEvent

    data class ClickItem(
        val entityType: MusicBrainzEntityType,
        val id: String,
    ) : AllLocalEntitiesUiEvent
}
