package ly.david.musicsearch.ui.common.recording

import androidx.compose.foundation.clickable
import androidx.compose.foundation.combinedClickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.material3.Icon
import androidx.compose.material3.ListItem
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import ly.david.musicsearch.shared.domain.common.ifNotNullOrEmpty
import ly.david.musicsearch.shared.domain.common.toDisplayTime
import ly.david.musicsearch.shared.domain.listitem.RecordingListItemModel
import ly.david.musicsearch.shared.domain.network.MusicBrainzEntityType
import ly.david.musicsearch.ui.common.getIcon
import ly.david.musicsearch.ui.common.icon.AddToCollectionIconButton
import ly.david.musicsearch.ui.common.icons.CustomIcons
import ly.david.musicsearch.ui.common.icons.Headphones
import ly.david.musicsearch.ui.common.icons.MusicVideo
import ly.david.musicsearch.ui.common.image.ThumbnailImage
import ly.david.musicsearch.ui.common.listitem.listItemColors
import ly.david.musicsearch.ui.common.locale.getAnnotatedName
import ly.david.musicsearch.ui.common.text.TextWithIcon
import ly.david.musicsearch.ui.common.text.fontWeight
import ly.david.musicsearch.ui.common.theme.LocalStrings
import ly.david.musicsearch.ui.common.theme.TINY_ICON_SIZE
import ly.david.musicsearch.ui.common.theme.TextStyles
import ly.david.musicsearch.ui.common.track.TrackListItem

/**
 * Also see [TrackListItem].
 */
@Composable
fun RecordingListItem(
    recording: RecordingListItemModel,
    modifier: Modifier = Modifier,
    onRecordingClick: RecordingListItemModel.() -> Unit = {},
    isSelected: Boolean = false,
    onSelect: (String) -> Unit = {},
    onEditCollectionClick: (String) -> Unit = {},
) {
    ListItem(
        headlineContent = {
            Text(
                text = recording.getAnnotatedName(),
                style = TextStyles.getCardBodyTextStyle(),
                fontWeight = recording.fontWeight,
            )
        },
        modifier = modifier.combinedClickable(
            onClick = { onRecordingClick(recording) },
            onLongClick = { onSelect(recording.id) },
        ),
        colors = listItemColors(isSelected = isSelected),
        supportingContent = {
            Column {
                Row(
                    modifier = Modifier.padding(top = 4.dp),
                    verticalAlignment = Alignment.CenterVertically,
                ) {
                    val dateAndLength = listOfNotNull(
                        recording.firstReleaseDate.takeIf { it.isNotEmpty() },
                        recording.length.toDisplayTime(),
                    ).joinToString("・")
                    Text(
                        text = dateAndLength,
                        style = TextStyles.getCardBodySubTextStyle(),
                        fontWeight = recording.fontWeight,
                    )

                    if (recording.video) {
                        Icon(
                            imageVector = CustomIcons.MusicVideo,
                            modifier = Modifier
                                .padding(start = 4.dp)
                                .size(TINY_ICON_SIZE.dp),
                            contentDescription = LocalStrings.current.video,
                        )
                    }
                }

                recording.formattedArtistCredits.ifNotNullOrEmpty {
                    Text(
                        text = it,
                        modifier = Modifier.padding(top = 4.dp),
                        style = TextStyles.getCardBodySubTextStyle(),
                        fontWeight = recording.fontWeight,
                    )
                }

                val listenCount = recording.listenCount
                if (listenCount != null) {
                    TextWithIcon(
                        modifier = Modifier.padding(top = 4.dp),
                        imageVector = CustomIcons.Headphones,
                        text = listenCount.toString(),
                        iconSize = TINY_ICON_SIZE,
                        textStyle = TextStyles.getCardBodySubTextStyle(),
                    )
                    // TODO: show relatively how often you've listened to a recording compared to other recordings
                    //  by a browse method
                    //  Each details entity that can show a list of recordings need to fetch their most listened recording
                    //  and pass it here.
//                    LinearProgressIndicator(
//                        progress = { listenCount / maxOf(track.mostListenedTrackCount, 1).toFloat() },
//                        modifier = Modifier
//                            .height(4.dp)
//                            .fillMaxWidth(),
//                        color = MaterialTheme.colorScheme.primary,
//                        trackColor = Color.Transparent,
//                    )
                }
            }
        },
        leadingContent = {
            ThumbnailImage(
                url = "",
                imageId = null,
                placeholderIcon = MusicBrainzEntityType.RECORDING.getIcon(),
                modifier = Modifier
                    .clickable {
                        onSelect(recording.id)
                    },
                isSelected = isSelected,
            )
        },
        trailingContent = {
            AddToCollectionIconButton(
                entityListItemModel = recording,
                onClick = onEditCollectionClick,
            )
        },
    )
}
