package ly.david.musicsearch.shared.feature.details.utils

import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.pager.HorizontalPager
import androidx.compose.foundation.pager.PagerState
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.TopAppBarScrollBehavior
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.nestedscroll.nestedScroll
import ly.david.musicsearch.shared.domain.details.MusicBrainzDetailsModel
import ly.david.musicsearch.shared.domain.list.SortOption
import ly.david.musicsearch.shared.domain.list.showTypes
import ly.david.musicsearch.shared.domain.network.MusicBrainzEntityType
import ly.david.musicsearch.shared.feature.details.release.TracksByReleaseUi
import ly.david.musicsearch.ui.common.fullscreen.DetailsWithErrorHandling
import ly.david.musicsearch.ui.common.paging.EntitiesLazyPagingItems
import ly.david.musicsearch.ui.common.paging.EntitiesPagingListUi
import ly.david.musicsearch.ui.common.paging.EntitiesPagingListUiState
import ly.david.musicsearch.ui.common.paging.getLoadedIdsForTab
import ly.david.musicsearch.ui.common.topappbar.Tab
import ly.david.musicsearch.ui.common.topappbar.toMusicBrainzEntityType

@OptIn(ExperimentalMaterial3Api::class)
@Composable
internal fun <T : MusicBrainzDetailsModel> DetailsHorizontalPager(
    pagerState: PagerState,
    state: DetailsUiState<T>,
    entitiesLazyPagingItems: EntitiesLazyPagingItems,
    innerPadding: PaddingValues,
    scrollBehavior: TopAppBarScrollBehavior,
    onEditCollectionClick: (String) -> Unit = {},
    requestForMissingCoverArtUrl: (entityId: String, entity: MusicBrainzEntityType?) -> Unit = { _, _ -> },
    detailsScreen: @Composable ((T) -> Unit),
) {
    val eventSink = state.eventSink

    HorizontalPager(
        state = pagerState,
        modifier = Modifier
            .padding(innerPadding)
            .nestedScroll(scrollBehavior.nestedScrollConnection),
    ) { page ->
        val tab = state.tabs[page]

        when (tab) {
            Tab.DETAILS -> {
                DetailsWithErrorHandling(
                    modifier = Modifier.fillMaxSize(),
                    isLoading = state.detailsTabUiState.isLoading,
                    handledException = state.detailsTabUiState.handledException,
                    onRefresh = {
                        eventSink(DetailsUiEvent.ForceRefreshDetails)
                    },
                    detailsModel = state.detailsModel,
                ) { detailsModel ->
                    detailsScreen(detailsModel)
                }
            }

            Tab.TRACKS -> {
                TracksByReleaseUi(
                    uiState = state.allEntitiesListUiState.tracksByReleaseUiState,
                    onRecordingClick = { id ->
                        eventSink(
                            DetailsUiEvent.ClickItem(
                                entityType = MusicBrainzEntityType.RECORDING,
                                id = id,
                            ),
                        )
                    },
                )
            }

            else -> {
                val uiState = when (tab) {
                    Tab.ARTISTS -> EntitiesPagingListUiState(
                        lazyPagingItems = entitiesLazyPagingItems.artistsLazyPagingItems,
                        lazyListState = state.allEntitiesListUiState.artistsListUiState.lazyListState,
                    )

                    Tab.EVENTS -> EntitiesPagingListUiState(
                        lazyPagingItems = entitiesLazyPagingItems.eventsLazyPagingItems,
                        lazyListState = state.allEntitiesListUiState.eventsListUiState.lazyListState,
                    )

                    Tab.LABELS -> EntitiesPagingListUiState(
                        lazyPagingItems = entitiesLazyPagingItems.labelsLazyPagingItems,
                        lazyListState = state.allEntitiesListUiState.labelsListUiState.lazyListState,
                    )

                    Tab.PLACES -> EntitiesPagingListUiState(
                        lazyPagingItems = entitiesLazyPagingItems.placesLazyPagingItems,
                        lazyListState = state.allEntitiesListUiState.placesListUiState.lazyListState,
                    )

                    Tab.RECORDINGS -> EntitiesPagingListUiState(
                        lazyPagingItems = entitiesLazyPagingItems.recordingsLazyPagingItems,
                        lazyListState = state.allEntitiesListUiState.recordingsListUiState.lazyListState,
                    )

                    Tab.RELEASES -> EntitiesPagingListUiState(
                        lazyPagingItems = entitiesLazyPagingItems.releasesLazyPagingItems,
                        lazyListState = state.allEntitiesListUiState.releasesListUiState.lazyListState,
                        showMoreInfo = (
                            state.allEntitiesListUiState.releasesListUiState.sortOption as? SortOption.Release
                            )?.showMoreInfo == true,
                    )

                    Tab.RELEASE_GROUPS -> EntitiesPagingListUiState(
                        lazyPagingItems = entitiesLazyPagingItems.releaseGroupsLazyPagingItems,
                        lazyListState = state.allEntitiesListUiState.releaseGroupsListUiState.lazyListState,
                        showMoreInfo = state.allEntitiesListUiState.releaseGroupsListUiState.sortOption.showTypes(),
                    )

                    Tab.WORKS -> EntitiesPagingListUiState(
                        lazyPagingItems = entitiesLazyPagingItems.worksLazyPagingItems,
                        lazyListState = state.allEntitiesListUiState.worksListUiState.lazyListState,
                    )

                    Tab.RELATIONSHIPS -> EntitiesPagingListUiState(
                        lazyPagingItems = entitiesLazyPagingItems.relationsLazyPagingItems,
                        lazyListState = state.allEntitiesListUiState.relationsUiState.lazyListState,
                    )

                    else -> {
                        error("$tab tab should not be accessible for ${state.browseMethod.entityType}.")
                    }
                }
                val tabEntity = tab.toMusicBrainzEntityType()
                EntitiesPagingListUi(
                    uiState = uiState,
                    now = state.detailsTabUiState.now,
                    onItemClick = { entity, id ->
                        eventSink(
                            DetailsUiEvent.ClickItem(
                                entityType = entity,
                                id = id,
                            ),
                        )
                    },
                    selectedIds = state.selectionState.selectedIds,
                    onSelect = {
                        eventSink(
                            DetailsUiEvent.ToggleSelectItem(
                                collectableId = it,
                                loadedCount = entitiesLazyPagingItems.getLoadedIdsForTab(
                                    tab = tab,
                                ).size,
                            ),
                        )
                    },
                    onEditCollectionClick = onEditCollectionClick,
                    requestForMissingCoverArtUrl = { entityId ->
                        requestForMissingCoverArtUrl(entityId, tabEntity)
                    },
                )
            }
        }
    }
}
