package io.github.dorumrr.happytaxes

import android.os.Bundle
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.fragment.app.FragmentActivity
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Assessment
import androidx.compose.material.icons.filled.Home
import androidx.compose.material.icons.filled.Lock
import androidx.compose.material.icons.filled.Settings
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.NavigationBar
import androidx.compose.material3.NavigationBarItem
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.produceState
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.navigation.NavDestination.Companion.hierarchy
import androidx.navigation.compose.currentBackStackEntryAsState
import androidx.navigation.compose.rememberNavController
import dagger.hilt.android.AndroidEntryPoint
import dagger.hilt.android.EntryPointAccessors
import io.github.dorumrr.happytaxes.data.repository.PreferencesRepository
import io.github.dorumrr.happytaxes.di.RepositoryEntryPoint
import io.github.dorumrr.happytaxes.ui.navigation.NavGraph
import io.github.dorumrr.happytaxes.ui.navigation.Screen
import io.github.dorumrr.happytaxes.ui.theme.HappyTaxesTheme
import io.github.dorumrr.happytaxes.util.BiometricAuthManager
import kotlinx.coroutines.flow.first
import javax.inject.Inject

/**
 * Main activity for HappyTaxes.
 *
 * Entry point for the app, hosts the Compose UI with navigation.
 * Checks onboarding status and sets start destination accordingly.
 * Includes Material 3 bottom navigation bar for main screens.
 */
@AndroidEntryPoint
class MainActivity : FragmentActivity() {

    @Inject
    lateinit var biometricAuthManager: BiometricAuthManager

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()
        setContent {
            val context = LocalContext.current
            val navController = rememberNavController()

            // Get repositories from Hilt
            val entryPoint = EntryPointAccessors.fromApplication(
                context.applicationContext,
                RepositoryEntryPoint::class.java
            )
            val preferencesRepository = entryPoint.preferencesRepository()
            val profileRepository = entryPoint.profileRepository()

            // Get dynamic color preference for theme
            val dynamicColorEnabled by preferencesRepository.getDynamicColorEnabled()
                .collectAsState(initial = false)

            // Get biometric auth preference
            // Initial value is true (assume locked) for security - prevents content flash before preference loads
            val biometricAuthEnabled by preferencesRepository.getBiometricAuthEnabled()
                .collectAsState(initial = true)

            // Get auto-lock timeout preference
            val autoLockTimeout by preferencesRepository.getAutoLockTimeout()
                .collectAsState(initial = 60)

            // Track authentication state
            var isAuthenticated by remember { mutableStateOf(false) }
            var authenticationAttempted by remember { mutableStateOf(false) }
            var backgroundTimestamp by remember { mutableStateOf<Long?>(null) }

            // Function to show authentication prompt
            val showAuthenticationPrompt = {
                authenticationAttempted = true
                android.util.Log.d("MainActivity", "Showing biometric authentication prompt")
                biometricAuthManager.authenticate(
                    activity = this@MainActivity,
                    title = "Unlock HappyTaxes",
                    subtitle = "Authenticate to access your data",
                    description = null,
                    onSuccess = {
                        android.util.Log.d("MainActivity", "Authentication successful")
                        isAuthenticated = true
                        backgroundTimestamp = null
                    },
                    onError = { error ->
                        android.util.Log.e("MainActivity", "Authentication error: $error")
                        // Keep showing prompt until authenticated
                        authenticationAttempted = false
                    },
                    onCancel = {
                        android.util.Log.d("MainActivity", "Authentication cancelled - exiting app")
                        // User cancelled - exit app
                        finish()
                    }
                )
            }

            // Show biometric prompt on launch if enabled
            LaunchedEffect(biometricAuthEnabled) {
                android.util.Log.d("MainActivity", "LaunchedEffect: biometricAuthEnabled=$biometricAuthEnabled, authenticationAttempted=$authenticationAttempted")
                if (biometricAuthEnabled && !authenticationAttempted) {
                    showAuthenticationPrompt()
                } else if (!biometricAuthEnabled) {
                    android.util.Log.d("MainActivity", "Biometric auth not enabled - allowing access")
                    // Auth not required, allow access
                    isAuthenticated = true
                }
            }

            // Handle lifecycle events for auto-lock
            DisposableEffect(Unit) {
                val lifecycleObserver = object : DefaultLifecycleObserver {
                    override fun onStop(owner: LifecycleOwner) {
                        // App going to background
                        if (biometricAuthEnabled && isAuthenticated) {
                            backgroundTimestamp = System.currentTimeMillis()
                            android.util.Log.d("MainActivity", "App going to background at $backgroundTimestamp")
                        }
                    }

                    override fun onStart(owner: LifecycleOwner) {
                        // App coming to foreground
                        if (biometricAuthEnabled && isAuthenticated && backgroundTimestamp != null) {
                            val elapsedSeconds = (System.currentTimeMillis() - backgroundTimestamp!!) / 1000
                            android.util.Log.d("MainActivity", "App coming to foreground. Elapsed: ${elapsedSeconds}s, Timeout: ${autoLockTimeout}s")

                            if (elapsedSeconds >= autoLockTimeout) {
                                android.util.Log.d("MainActivity", "Auto-lock timeout exceeded - requiring authentication")
                                isAuthenticated = false
                                authenticationAttempted = false
                                showAuthenticationPrompt()
                            } else {
                                android.util.Log.d("MainActivity", "Within timeout - no authentication required")
                            }
                        }
                    }
                }

                lifecycle.addObserver(lifecycleObserver)

                onDispose {
                    lifecycle.removeObserver(lifecycleObserver)
                }
            }

            HappyTaxesTheme(dynamicColor = dynamicColorEnabled) {
                Surface(
                    modifier = Modifier.fillMaxSize(),
                    color = MaterialTheme.colorScheme.background
                ) {
                    // Show lock screen when authentication is required but not yet completed
                    if (!isAuthenticated && biometricAuthEnabled) {
                        // Privacy screen - covers content while biometric dialog is shown
                        Box(
                            modifier = Modifier
                                .fillMaxSize()
                                .background(MaterialTheme.colorScheme.background)
                                .padding(top = 120.dp),
                            contentAlignment = Alignment.TopCenter
                        ) {
                            Column(
                                horizontalAlignment = Alignment.CenterHorizontally
                            ) {
                                Icon(
                                    imageVector = Icons.Default.Lock,
                                    contentDescription = "Locked",
                                    modifier = Modifier.size(64.dp),
                                    tint = MaterialTheme.colorScheme.primary
                                )
                                Spacer(modifier = Modifier.height(16.dp))
                                Text(
                                    text = "HappyTaxes",
                                    style = MaterialTheme.typography.headlineMedium,
                                    color = MaterialTheme.colorScheme.onBackground
                                )
                                Spacer(modifier = Modifier.height(8.dp))
                                Text(
                                    text = "Authenticate to continue",
                                    style = MaterialTheme.typography.bodyMedium,
                                    color = MaterialTheme.colorScheme.onSurfaceVariant
                                )
                            }
                        }
                    } else if (isAuthenticated) {
                        // Only show app content when authenticated (or auth not required)
                        // Wait for DataStore to load before determining start destination
                        val startDestination by produceState<String?>(initialValue = null) {
                        val onboardingComplete = preferencesRepository.isOnboardingComplete().first()

                        value = when {
                            // Not onboarded yet -> go to onboarding
                            !onboardingComplete -> Screen.Onboarding.route

                            // Onboarded -> go directly to ledger (last used profile will be loaded)
                            else -> Screen.Ledger.route
                        }
                    }

                    // Track current navigation destination
                    val navBackStackEntry by navController.currentBackStackEntryAsState()
                    val currentDestination = navBackStackEntry?.destination

                    // Define bottom nav screens
                    val bottomNavScreens = listOf(
                        Screen.Ledger.route,
                        Screen.Reports.route,
                        Screen.Settings.route
                    )

                    // Show bottom nav only on main screens (not onboarding, detail screens, etc.)
                    val showBottomNav = currentDestination?.route in bottomNavScreens

                    // Only show NavGraph after start destination is determined
                    startDestination?.let { destination ->
                        Scaffold(
                            contentWindowInsets = WindowInsets(0, 0, 0, 0),
                            bottomBar = {
                                if (showBottomNav) {
                                    NavigationBar {
                                        // Transactions
                                        NavigationBarItem(
                                            icon = { Icon(Icons.Default.Home, contentDescription = null) },
                                            label = { Text("Transactions") },
                                            selected = currentDestination?.hierarchy?.any { it.route == Screen.Ledger.route } == true,
                                            onClick = {
                                                if (currentDestination?.route != Screen.Ledger.route) {
                                                    navController.navigate(Screen.Ledger.route) {
                                                        // Pop up to start destination to avoid building up back stack
                                                        popUpTo(Screen.Ledger.route) {
                                                            saveState = true
                                                        }
                                                        launchSingleTop = true
                                                        restoreState = true
                                                    }
                                                }
                                            }
                                        )

                                        // Reports
                                        NavigationBarItem(
                                            icon = { Icon(Icons.Default.Assessment, contentDescription = null) },
                                            label = { Text("Reports") },
                                            selected = currentDestination?.hierarchy?.any { it.route == Screen.Reports.route } == true,
                                            onClick = {
                                                if (currentDestination?.route != Screen.Reports.route) {
                                                    navController.navigate(Screen.Reports.route) {
                                                        popUpTo(Screen.Ledger.route) {
                                                            saveState = true
                                                        }
                                                        launchSingleTop = true
                                                        restoreState = true
                                                    }
                                                }
                                            }
                                        )

                                        // Settings
                                        NavigationBarItem(
                                            icon = { Icon(Icons.Default.Settings, contentDescription = null) },
                                            label = { Text("Settings") },
                                            selected = currentDestination?.hierarchy?.any { it.route == Screen.Settings.route } == true,
                                            onClick = {
                                                if (currentDestination?.route != Screen.Settings.route) {
                                                    navController.navigate(Screen.Settings.route) {
                                                        popUpTo(Screen.Ledger.route) {
                                                            saveState = true
                                                        }
                                                        launchSingleTop = true
                                                        restoreState = true
                                                    }
                                                }
                                            }
                                        )
                                    }
                                }
                            }
                        ) { paddingValues ->
                            Box(modifier = Modifier.padding(paddingValues)) {
                                NavGraph(
                                    navController = navController,
                                    startDestination = destination
                                )
                            }
                        }
                    }
                    }
                }
            }
        }
    }
}

