package io.github.dorumrr.de1984.data.receiver

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.util.Log
import io.github.dorumrr.de1984.De1984Application
import io.github.dorumrr.de1984.data.service.NewAppNotificationManager
import io.github.dorumrr.de1984.domain.usecase.HandleNewAppInstallUseCase
import io.github.dorumrr.de1984.utils.Constants
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch


class PackageAddedReceiver : BroadcastReceiver() {

    companion object {
        private const val TAG = "PackageAddedReceiver"
    }

    override fun onReceive(context: Context, intent: Intent?) {
        try {
            // Initialize dependencies
            val app = context.applicationContext as De1984Application
            val handleNewAppInstallUseCase = app.dependencies.provideHandleNewAppInstallUseCase()
            val newAppNotificationManager = app.dependencies.newAppNotificationManager

            // Extract UID from intent FIRST for multi-user support
            // UID format: userId * 100000 + appId
            val uid = intent?.getIntExtra(Intent.EXTRA_UID, -1)?.takeIf { it >= 0 }
            val userId = uid?.let { it / 100000 } ?: 0

            val packageName = validateAndExtractPackageName(context, intent, userId)
            if (packageName == null) {
                return
            }

            if (!areNewAppNotificationsEnabled(context)) {
                return
            }

            // Use goAsync() to keep receiver alive while coroutine runs
            val pendingResult = goAsync()

            // Use app's coroutine scope instead of creating orphaned scope
            // This ensures proper cancellation and resource cleanup
            app.dependencies.applicationScope.launch(Dispatchers.IO) {
                try {
                    handleNewAppInstallUseCase.execute(packageName, uid)
                        .onSuccess {
                            newAppNotificationManager.showNewAppNotification(packageName)
                        }
                } catch (e: Exception) {
                    // Error processing new app
                } finally {
                    // Signal that async work is complete
                    pendingResult.finish()
                }
            }

        } catch (e: Exception) {
            // Error in PackageAddedReceiver
        }
    }
    
    private fun validateAndExtractPackageName(context: Context, intent: Intent?, userId: Int): String? {
        if (intent?.action != Intent.ACTION_PACKAGE_ADDED) {
            return null
        }

        val data = intent.data
        if (data == null || data.scheme != "package") {
            return null
        }

        val packageName = data.schemeSpecificPart
        if (packageName.isNullOrBlank()) {
            return null
        }

        if (!packageName.contains(".")) {
            return null
        }

        if (Constants.App.isOwnApp(packageName)) {
            return null
        }

        if (!isValidPackage(context, packageName, userId)) {
            return null
        }

        val isReplacing = intent.getBooleanExtra(Intent.EXTRA_REPLACING, false)
        if (isReplacing) {
            return null
        }

        return packageName
    }

    /**
     * Check if package exists using HiddenApiHelper for multi-user support.
     * Work profile apps are only visible when queried with the correct userId.
     */
    private fun isValidPackage(context: Context, packageName: String, userId: Int): Boolean {
        return try {
            io.github.dorumrr.de1984.data.multiuser.HiddenApiHelper.getApplicationInfoAsUser(
                context, packageName, 0, userId
            )
            true
        } catch (e: Exception) {
            false
        }
    }
    
    private fun areNewAppNotificationsEnabled(context: Context): Boolean {
        return try {
            val prefs = context.getSharedPreferences(Constants.Settings.PREFS_NAME, Context.MODE_PRIVATE)
            prefs.getBoolean(
                Constants.Settings.KEY_NEW_APP_NOTIFICATIONS,
                Constants.Settings.DEFAULT_NEW_APP_NOTIFICATIONS
            )
        } catch (e: Exception) {
            Constants.Settings.DEFAULT_NEW_APP_NOTIFICATIONS
        }
    }
}
