// Copyright (c) 2017, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.
@TestOn('vm')
library;

import 'dart:async';

import 'package:build/build.dart';
import 'package:build_runner/src/build/run_builder.dart';
import 'package:build_runner/src/build/single_step_reader_writer.dart';
import 'package:package_config/package_config_types.dart';
import 'package:test/test.dart';

import '../common/common.dart';

void main() {
  late InternalTestReaderWriter readerWriter;
  final primary = makeAssetId('a|web/primary.txt');
  final inputs = {primary: 'foo'};
  late Resource resource;
  late bool resourceDisposed;
  late Builder builder;

  setUp(() async {
    resourceDisposed = false;
    resource = Resource(
      () => 0,
      dispose: (_) {
        resourceDisposed = true;
      },
    );
    builder = TestBuilder(
      extraWork: (buildStep, _) => buildStep.fetchResource(resource),
    );
    readerWriter = InternalTestReaderWriter();
    addAssets(inputs, readerWriter);
  });

  group('Given a ResourceManager', () {
    late TrackingResourceManager resourceManager;

    setUp(() async {
      resourceManager = TrackingResourceManager();
      await runBuilder(
        builder,
        inputs.keys,
        SingleStepReaderWriter.fakeFor(readerWriter),
        null,
        resourceManager: resourceManager,
      );
    });

    tearDown(() async {
      await resourceManager.disposeAll();
    });

    test('uses the provided resource manager', () async {
      expect(resourceManager.fetchedResources, orderedEquals([resource]));
    });

    test('doesn\'t dispose the provided resource manager', () async {
      expect(resourceDisposed, false);
      expect(resourceManager.disposed, false);
      // sanity check
      await resourceManager.disposeAll();
      expect(resourceDisposed, true);
      expect(resourceManager.disposed, true);
    });
  });

  group('With a default ResourceManager', () {
    setUp(() async {
      await runBuilder(
        builder,
        inputs.keys,
        SingleStepReaderWriter.fakeFor(readerWriter),
        null,
      );
    });

    test('disposes the default resource manager', () async {
      expect(resourceDisposed, true);
    });
  });

  group('can resolve package config', () {
    setUp(() {
      readerWriter.testing.writeBytes(makeAssetId('build|lib/foo.txt'), [
        1,
        2,
        3,
      ]);

      builder = TestBuilder(
        extraWork: (buildStep, _) async {
          final config = await buildStep.packageConfig;

          final buildPackage = config.packages.singleWhere(
            (p) => p.name == 'build',
          );
          expect(buildPackage.root, Uri.parse('asset:build/'));
          expect(buildPackage.packageUriRoot, Uri.parse('asset:build/lib/'));
          expect(buildPackage.languageVersion, LanguageVersion(3, 7));

          final resolvedBuildUri =
              config.resolve(Uri.parse('package:build/foo.txt'))!;
          expect(
            await buildStep.canRead(AssetId.resolve(resolvedBuildUri)),
            isTrue,
          );
        },
      );
    });

    test('from default', () async {
      await runBuilder(
        builder,
        inputs.keys,
        SingleStepReaderWriter.fakeFor(readerWriter),
        null,
      );
    });

    test('when provided', () async {
      await runBuilder(
        builder,
        inputs.keys,
        SingleStepReaderWriter.fakeFor(readerWriter),
        null,
        packageConfig: PackageConfig([
          Package(
            'build',
            Uri.file('/foo/bar/'),
            languageVersion: LanguageVersion(3, 7),
          ),
        ]),
      );
    });
  });
}

class TrackingResourceManager extends ResourceManager {
  bool disposed = false;
  final fetchedResources = <Resource>{};

  @override
  Future<T> fetch<T>(Resource<T> resource) {
    fetchedResources.add(resource);
    return super.fetch(resource);
  }

  @override
  Future<void> disposeAll() {
    disposed = true;
    return super.disposeAll();
  }
}
