// Copyright (c) 2023, the Dart project authors. Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

import 'package:analyzer/src/dart/error/syntactic_errors.dart';
import 'package:analyzer/src/error/codes.dart';
import 'package:test_reflective_loader/test_reflective_loader.dart';

import 'context_collection_resolution.dart';

main() {
  defineReflectiveSuite(() {
    defineReflectiveTests(WhileStatementResolutionTest);
  });
}

@reflectiveTest
class WhileStatementResolutionTest extends PubPackageResolutionTest {
  test_break() async {
    await assertNoErrorsInCode('''
void f() {
  while (true) {
    break;
  }
}
''');

    var node = findNode.singleWhileStatement;
    assertResolvedNodeText(node, r'''
WhileStatement
  whileKeyword: while
  leftParenthesis: (
  condition: BooleanLiteral
    literal: true
    staticType: bool
  rightParenthesis: )
  body: Block
    leftBracket: {
    statements
      BreakStatement
        breakKeyword: break
        semicolon: ;
    rightBracket: }
''');
  }

  test_break_label() async {
    await assertNoErrorsInCode('''
void f() {
  L: while (true) {
    break L;
  }
}
''');

    var node = findNode.singleLabeledStatement;
    assertResolvedNodeText(node, r'''
LabeledStatement
  labels
    Label
      label: SimpleIdentifier
        token: L
        element: L@13
        staticType: null
      colon: :
  statement: WhileStatement
    whileKeyword: while
    leftParenthesis: (
    condition: BooleanLiteral
      literal: true
      staticType: bool
    rightParenthesis: )
    body: Block
      leftBracket: {
      statements
        BreakStatement
          breakKeyword: break
          label: SimpleIdentifier
            token: L
            element: L@13
            staticType: null
          semicolon: ;
      rightBracket: }
''');
  }

  test_break_label_unresolved() async {
    await assertErrorsInCode(
      '''
void f() {
  while (true) {
    break L;
  }
}
''',
      [error(CompileTimeErrorCode.labelUndefined, 38, 1)],
    );

    var node = findNode.singleWhileStatement;
    assertResolvedNodeText(node, r'''
WhileStatement
  whileKeyword: while
  leftParenthesis: (
  condition: BooleanLiteral
    literal: true
    staticType: bool
  rightParenthesis: )
  body: Block
    leftBracket: {
    statements
      BreakStatement
        breakKeyword: break
        label: SimpleIdentifier
          token: L
          element: <null>
          staticType: null
        semicolon: ;
    rightBracket: }
''');
  }

  test_condition_super() async {
    await assertErrorsInCode(
      '''
class A {
  void f() {
    while (super) {}
  }
}
''',
      [
        error(ParserErrorCode.missingAssignableSelector, 34, 5),
        error(CompileTimeErrorCode.nonBoolCondition, 34, 5),
      ],
    );

    var node = findNode.singleWhileStatement;
    assertResolvedNodeText(node, r'''
WhileStatement
  whileKeyword: while
  leftParenthesis: (
  condition: SuperExpression
    superKeyword: super
    staticType: A
  rightParenthesis: )
  body: Block
    leftBracket: {
    rightBracket: }
''');
  }

  test_continue() async {
    await assertNoErrorsInCode('''
void f() {
  while (true) {
    continue;
  }
}
''');

    var node = findNode.singleWhileStatement;
    assertResolvedNodeText(node, r'''
WhileStatement
  whileKeyword: while
  leftParenthesis: (
  condition: BooleanLiteral
    literal: true
    staticType: bool
  rightParenthesis: )
  body: Block
    leftBracket: {
    statements
      ContinueStatement
        continueKeyword: continue
        semicolon: ;
    rightBracket: }
''');
  }

  test_continue_label() async {
    await assertNoErrorsInCode('''
void f() {
  L: while (true) {
    continue L;
  }
}
''');

    var node = findNode.singleLabeledStatement;
    assertResolvedNodeText(node, r'''
LabeledStatement
  labels
    Label
      label: SimpleIdentifier
        token: L
        element: L@13
        staticType: null
      colon: :
  statement: WhileStatement
    whileKeyword: while
    leftParenthesis: (
    condition: BooleanLiteral
      literal: true
      staticType: bool
    rightParenthesis: )
    body: Block
      leftBracket: {
      statements
        ContinueStatement
          continueKeyword: continue
          label: SimpleIdentifier
            token: L
            element: L@13
            staticType: null
          semicolon: ;
      rightBracket: }
''');
  }

  test_continue_label_unresolved() async {
    await assertErrorsInCode(
      '''
void f() {
  while (true) {
    continue L;
  }
}
''',
      [error(CompileTimeErrorCode.labelUndefined, 41, 1)],
    );

    var node = findNode.singleWhileStatement;
    assertResolvedNodeText(node, r'''
WhileStatement
  whileKeyword: while
  leftParenthesis: (
  condition: BooleanLiteral
    literal: true
    staticType: bool
  rightParenthesis: )
  body: Block
    leftBracket: {
    statements
      ContinueStatement
        continueKeyword: continue
        label: SimpleIdentifier
          token: L
          element: <null>
          staticType: null
        semicolon: ;
    rightBracket: }
''');
  }
}
