/*
 * This file is part of Satunes.
 *
 * Satunes is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software Foundation,
 * either version 3 of the License, or (at your option) any later version.
 * Satunes is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with Satunes.
 * If not, see <https://www.gnu.org/licenses/>.
 *
 * *** INFORMATION ABOUT THE AUTHOR *****
 * The author of this file is Antoine Pirlot, the owner of this project.
 * You find this original project on Codeberg.
 *
 * My Codeberg link is: https://codeberg.org/antoinepirlot
 * This current project's link is: https://codeberg.org/antoinepirlot/Satunes
 */

package io.github.antoinepirlot.satunes.database.models.comparators

import io.github.antoinepirlot.satunes.database.exceptions.NotInAlbumException
import io.github.antoinepirlot.satunes.database.models.media.Music

/**
 * @author Antoine Pirlot on 18/10/2024
 */
object MusicInAlbumComparator : MediaComparator<Music>() {
    /**
     * Compare 2 music's in the same album.
     * Assume it is used only in Album's music set
     *
     * If o1 album is not the same as o2's throws [NotInAlbumException]
     * If at least one cdTrackNumber is null or the o1's is greater than o2's returns 1
     * If the o1 track number is < of the o2's return -1
     * returns 0 in other cases.
     *
     * @param o1 [Music]
     * @param o2 [Music]
     *
     * @throws [NotInAlbumException] if o1 has not the same album as o2
     * @return [Int]: -1 or 0 or 1 as specified.
     */
    override fun compare(o1: Music, o2: Music): Int {
        var cmp: Int = 0
        if (o1.album != o2.album) throw NotInAlbumException()
        if (o1.cdTrackNumber != null && o2.cdTrackNumber == null) cmp = -1
        else if (o1.cdTrackNumber == null && o2.cdTrackNumber != null) cmp = 1
        else if (o1.cdTrackNumber == null || o2.cdTrackNumber == null) cmp =
            o1.compareTo(o2) //both are null
        else if (o1.cdTrackNumber > o2.cdTrackNumber) cmp = 1
        else if (o1.cdTrackNumber < o2.cdTrackNumber) cmp = -1
        return this.getFinalCmp(cmp = cmp)
    }

}