/*
 * This file is part of Satunes.
 *
 * Satunes is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software Foundation,
 * either version 3 of the License, or (at your option) any later version.
 * Satunes is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with Satunes.
 * If not, see <https://www.gnu.org/licenses/>.
 *
 * *** INFORMATION ABOUT THE AUTHOR *****
 * The author of this file is Antoine Pirlot, the owner of this project.
 * You find this original project on Codeberg.
 *
 * My Codeberg link is: https://codeberg.org/antoinepirlot
 * This current project's link is: https://codeberg.org/antoinepirlot/Satunes
 */

package io.github.antoinepirlot.satunes.ui.components.forms

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.selection.selectableGroup
import androidx.compose.material3.Button
import androidx.compose.material3.SnackbarHostState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.MutableIntState
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.lifecycle.viewmodel.compose.viewModel
import io.github.antoinepirlot.jetpack_libs.components.texts.NormalText
import io.github.antoinepirlot.satunes.R
import io.github.antoinepirlot.satunes.data.local.LocalMainScope
import io.github.antoinepirlot.satunes.data.local.LocalSnackBarHostState
import io.github.antoinepirlot.satunes.data.states.PlaybackUiState
import io.github.antoinepirlot.satunes.data.viewmodels.PlaybackViewModel
import io.github.antoinepirlot.satunes.models.Timer
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

/**
 * @author Antoine Pirlot on 14/10/2024
 */

//max 8 hours
private const val MAX_HOURS: Int = 8
private const val MAX_MINUTES: Int = MAX_HOURS * 60
private const val MAX_SECONDS: Int = MAX_MINUTES * 60

@Composable
internal fun CreateTimerForm(
    modifier: Modifier = Modifier,
    playbackViewModel: PlaybackViewModel = viewModel(),
    onFinished: (() -> Unit)? = null,
) {
    val playbackUiState: PlaybackUiState by playbackViewModel.uiState.collectAsState()
    val timer: Timer? = playbackUiState.timer
    val isTimerRunning = timer != null
    val remainingTime: Long = playbackUiState.timerRemainingTime

    val secondsIntField: MutableIntState = rememberSaveable { mutableIntStateOf(0) }
    val minutesIntField: MutableIntState = rememberSaveable { mutableIntStateOf(0) }
    val hoursIntField: MutableIntState = rememberSaveable { mutableIntStateOf(0) }

    var job: Job? = null
    LaunchedEffect(remainingTime, timer) {
        secondsIntField.intValue = (timer?.getRemainingSeconds() ?: 0) % 60
        minutesIntField.intValue = (timer?.getRemainingMinutes() ?: 0) % 60
        hoursIntField.intValue = timer?.getRemainingHours() ?: 0
        job?.cancel()
        job = CoroutineScope(Dispatchers.IO).launch {
            delay(1000) //prevent system refreshing each ms for better performance
            playbackViewModel.refreshRemainingTime()
        }
    }

    Column(
        modifier = modifier
            .selectableGroup()
            .fillMaxWidth(),
        horizontalAlignment = Alignment.CenterHorizontally,
    ) {
        TimerFields(
            isTimerRunning = isTimerRunning,
            secondsIntField = secondsIntField,
            minutesIntField = minutesIntField,
            hoursIntField = hoursIntField
        )

        Spacer(modifier = Modifier.size(16.dp))

        TimerButtons(
            onFinished = onFinished,
            isTimerRunning = isTimerRunning,
            secondsIntField = secondsIntField,
            minutesIntField = minutesIntField,
            hoursIntField = hoursIntField
        )
    }
}

@Composable
private fun TimerFields(
    modifier: Modifier = Modifier,
    isTimerRunning: Boolean,
    secondsIntField: MutableIntState,
    minutesIntField: MutableIntState,
    hoursIntField: MutableIntState
) {
    Box(
        modifier = modifier.fillMaxWidth(),
        contentAlignment = Alignment.Center,
    ) {
        Row(
            modifier = Modifier.fillMaxWidth(),
        ) {
            Field(
                modifier = Modifier.fillMaxWidth(fraction = 1f / 3f),
                enabled = !isTimerRunning,
                farLeft = true,
                value = hoursIntField,
                label = stringResource(R.string.hours_text_field_label),
                maxValue = MAX_HOURS
            )
            Field(
                modifier = Modifier.fillMaxWidth(fraction = 1f / 2f),
                enabled = !isTimerRunning,
                value = minutesIntField,
                label = stringResource(R.string.minutes_text_field_label),
                maxValue = MAX_MINUTES
            )
            Field(
                modifier = Modifier.fillMaxWidth(),
                enabled = !isTimerRunning,
                farRight = true,
                value = secondsIntField,
                label = stringResource(R.string.seconds_text_field_label),
                maxValue = MAX_SECONDS
            )
        }
    }
}

@Composable
private fun TimerButtons(
    modifier: Modifier = Modifier,
    playbackViewModel: PlaybackViewModel = viewModel(),
    onFinished: (() -> Unit)? = null,
    isTimerRunning: Boolean,
    secondsIntField: MutableIntState,
    minutesIntField: MutableIntState,
    hoursIntField: MutableIntState
) {
    val scope: CoroutineScope = LocalMainScope.current
    val snackBarHostState: SnackbarHostState = LocalSnackBarHostState.current

    Row(
        modifier = modifier.fillMaxWidth(),
        horizontalArrangement = Arrangement.Center
    ) {
        if (isTimerRunning) {
            Button(
                onClick = {
                    playbackViewModel.cancelTimer(
                        scope = scope,
                        snackBarHostState = snackBarHostState,
                    )
                    secondsIntField.intValue = 0
                    minutesIntField.intValue = 0
                    hoursIntField.intValue = 0
                    onFinished?.invoke()
                }
            ) {
                NormalText(text = stringResource(R.string.cancel))
            }

            Spacer(modifier = Modifier.size(5.dp))
        } else {
            Button(
                onClick = {
                    computeTime(
                        secondsIntField = secondsIntField,
                        minutesIntField = minutesIntField,
                        hoursIntField = hoursIntField
                    )
                    playbackViewModel.setTimer(
                        scope = scope,
                        snackBarHostState = snackBarHostState,
                        hours = hoursIntField.intValue,
                        minutes = minutesIntField.intValue,
                        seconds = secondsIntField.intValue
                    )
                    onFinished?.invoke()
                }
            ) {

                NormalText(text = stringResource(R.string.start_timer_button_content))
            }
        }
    }
}

private fun computeTime(
    secondsIntField: MutableIntState,
    minutesIntField: MutableIntState,
    hoursIntField: MutableIntState
) {
    val exceedMinutes = secondsIntField.intValue / 60
    minutesIntField.intValue += exceedMinutes
    val exceedHours = minutesIntField.intValue / 60
    hoursIntField.intValue += exceedHours
    if (hoursIntField.intValue >= MAX_HOURS) {
        hoursIntField.intValue = MAX_HOURS
        minutesIntField.intValue = 0
        secondsIntField.intValue = 0
    } else {
        secondsIntField.intValue -= exceedMinutes * 60
        minutesIntField.intValue -= exceedHours * 60
    }
}

@Preview
@Composable
private fun CreateTimerFormPreview() {
    CreateTimerForm()
}