/*
 * This file is part of Satunes.
 *
 * Satunes is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software Foundation,
 * either version 3 of the License, or (at your option) any later version.
 * Satunes is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with Satunes.
 * If not, see <https://www.gnu.org/licenses/>.
 *
 * *** INFORMATION ABOUT THE AUTHOR *****
 * The author of this file is Antoine Pirlot, the owner of this project.
 * You find this original project on Codeberg.
 *
 * My Codeberg link is: https://codeberg.org/antoinepirlot
 * This current project's link is: https://codeberg.org/antoinepirlot/Satunes
 */

package io.github.antoinepirlot.satunes.internet.updates

import android.annotation.SuppressLint
import android.app.DownloadManager
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.net.Uri
import android.os.Build
import android.os.Environment
import androidx.annotation.RequiresApi
import androidx.core.content.getSystemService
import io.github.antoinepirlot.satunes.internet.R
import io.github.antoinepirlot.satunes.utils.logger.SatunesLogger
import io.github.antoinepirlot.satunes.utils.utils.showToastOnUiThread

/**
 * @author Antoine Pirlot on 14/04/2024
 */

@RequiresApi(Build.VERSION_CODES.M)
object UpdateDownloadManager {
    private var downloadId: Long = -1
    private const val MIME_TYPE = "application/vnd.android.package-archive"
    private val _logger: SatunesLogger? = SatunesLogger.getLogger()

    fun downloadUpdateApk(context: Context) {
        if (UpdateCheckManager.downloadStatus.value == APKDownloadStatus.CHECKING || UpdateCheckManager.downloadStatus.value == APKDownloadStatus.DOWNLOADING) {
            return
        }
        UpdateCheckManager.downloadStatus.value = APKDownloadStatus.CHECKING
        try {
            if (UpdateCheckManager.updateAvailableStatus.value != UpdateAvailableStatus.AVAILABLE) {
                //Can't be downloaded
                UpdateCheckManager.downloadStatus.value = APKDownloadStatus.NOT_FOUND
                showToastOnUiThread(
                    context = context,
                    message = context.getString(R.string.download_not_found)
                )
                _logger?.warning("UpdateCheckManager.updateAvailableStatus.value != UpdateAvailableStatus.AVAILABLE")
                return
            }
            val downloadUrl: String = getDownloadUrl()
            val appName: String = downloadUrl.split("/").last()
            val downloadManager: DownloadManager = context.getSystemService()!!
            val downloadUri: Uri = Uri.parse(downloadUrl)
            val req: DownloadManager.Request = DownloadManager.Request(downloadUri)
            req.setMimeType(MIME_TYPE)
            val destination =
                "file://" + context.getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS)
                    .toString() + "/$appName"
            val destinationUri: Uri = Uri.parse(destination)
            req.setDestinationUri(destinationUri)

            req.setNotificationVisibility(DownloadManager.Request.VISIBILITY_VISIBLE)
            setDownloadReceiver(context = context)
            downloadId = downloadManager.enqueue(req)
            UpdateCheckManager.downloadStatus.value = APKDownloadStatus.DOWNLOADING
        } catch (e: Throwable) {
            UpdateCheckManager.downloadStatus.value = APKDownloadStatus.FAILED
            _logger?.severe(e.message)
            UpdateCheckManager.updateAvailableStatus.value = UpdateAvailableStatus.UNDEFINED
            throw e
        }
    }

    /**
     * Get the download url for the latest version. It looks like:
     * "https://github.com/antoinepirlot/Satunes/releases/download/vx.y.z-beta/[app name]_vx.y.z[-versionType].apk"
     *
     * @return the download url or null if not found
     */
    private fun getDownloadUrl(): String {
        val latestVersion: String = UpdateCheckManager.latestVersion.value!!
        return "https://codeberg.org/antoinepirlot/Satunes/releases/download/$latestVersion/Satunes_$latestVersion.apk"
    }

    @SuppressLint("UnspecifiedRegisterReceiverFlag")
    private fun setDownloadReceiver(context: Context) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            context.registerReceiver(
                DownloadReceiver, IntentFilter(DownloadManager.ACTION_DOWNLOAD_COMPLETE),
                Context.RECEIVER_EXPORTED
            )
        } else
            context.registerReceiver(
                DownloadReceiver,
                IntentFilter(DownloadManager.ACTION_DOWNLOAD_COMPLETE)
            )
    }

    /**
     * Launch the installation procedure by request the user to install the app.
     */
    fun installUpdate(context: Context) {
        try {
            val downloadManager: DownloadManager =
                context.getSystemService(Context.DOWNLOAD_SERVICE) as DownloadManager
            val contentUri: Uri = downloadManager.getUriForDownloadedFile(downloadId)
            val install = Intent(Intent.ACTION_VIEW)
            install.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
            install.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP)
            install.putExtra(Intent.EXTRA_NOT_UNKNOWN_SOURCE, true)
            install.data = contentUri
            context.startActivity(install)
        } catch (e: Throwable) {
            _logger?.severe(e.message)
            throw e
        }
    }
}