/*
 * This file is part of Satunes.
 *
 * Satunes is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software Foundation,
 * either version 3 of the License, or (at your option) any later version.
 *
 * Satunes is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See the GNU General Public License for more details.
 *  You should have received a copy of the GNU General Public License along with Satunes.
 *
 * If not, see <https://www.gnu.org/licenses/>.
 *
 * **** INFORMATION ABOUT THE AUTHOR *****
 * The author of this file is Antoine Pirlot, the owner of this project.
 * You find this original project on Codeberg.
 *
 * My Codeberg link is: https://codeberg.org/antoinepirlot
 * This current project's link is: https://codeberg.org/antoinepirlot/Satunes
 */

package io.github.antoinepirlot.satunes.database.services.settings

import android.content.Context
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.mutableStateOf
import androidx.datastore.preferences.core.MutablePreferences
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.booleanPreferencesKey
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.stringPreferencesKey
import io.github.antoinepirlot.satunes.database.models.UpdateChannel
import io.github.antoinepirlot.satunes.database.services.settings.SettingsManager.dataStore
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.map

/**
 * Settings related to UI of Satunes
 * @author Antoine Pirlot 03/02/2025
 */
internal object SatunesSettings {

    // DEFAULT VALUES

    private const val DEFAULT_WHATS_NEW_SEEN: Boolean = false
    private const val DEFAULT_WHATS_NEW_VERSION_SEEN: String = ""
    private const val DEFAULT_LOGS_ACTIVATION: Boolean = true
    private const val DEFAULT_INCLUDE_EXCLUDE_SEEN: Boolean = false
    private val DEFAULT_UPDATE_CHANNEL: UpdateChannel = UpdateChannel.STABLE

    // KEYS

    private val WHATS_NEW_SEEN_KEY: Preferences.Key<Boolean> =
        booleanPreferencesKey("whats_new_seen")
    private val WHATS_NEW_VERSION_SEEN_KEY: Preferences.Key<String> =
        stringPreferencesKey("whats_new_version_seen")
    private val LOGS_ACTIVATION_KEY: Preferences.Key<Boolean> =
        booleanPreferencesKey("logs_activation")
    private val INCLUDE_EXCLUDE_SEEN_KEY: Preferences.Key<Boolean> =
        booleanPreferencesKey("see_include_exclude")
    private val UPDATE_CHANNEL_KEY: Preferences.Key<String> = stringPreferencesKey("update_channel")

    // VARIABLES

    var whatsNewSeen: Boolean = DEFAULT_WHATS_NEW_SEEN
        private set
    private var whatsNewVersionSeen: String = DEFAULT_WHATS_NEW_VERSION_SEEN
    var logsActivation: MutableState<Boolean> = mutableStateOf(DEFAULT_LOGS_ACTIVATION)
        private set
    var includeExcludeSeen: Boolean = DEFAULT_INCLUDE_EXCLUDE_SEEN
    var updateChannel: MutableState<UpdateChannel> = mutableStateOf(DEFAULT_UPDATE_CHANNEL)

    suspend fun loadSettings(context: Context) {
        context.dataStore.data.map { preferences: Preferences ->
            this.whatsNewSeen = preferences[WHATS_NEW_SEEN_KEY] ?: DEFAULT_WHATS_NEW_SEEN
            this.whatsNewVersionSeen =
                preferences[WHATS_NEW_VERSION_SEEN_KEY] ?: DEFAULT_WHATS_NEW_VERSION_SEEN
            this.logsActivation.value = preferences[LOGS_ACTIVATION_KEY] ?: DEFAULT_LOGS_ACTIVATION
            this.includeExcludeSeen =
                preferences[INCLUDE_EXCLUDE_SEEN_KEY] ?: DEFAULT_INCLUDE_EXCLUDE_SEEN

            val updateChannelName: String? = preferences[UPDATE_CHANNEL_KEY]
            if (updateChannelName == null) this.updateChannel.value = DEFAULT_UPDATE_CHANNEL
            else this.updateChannel.value = UpdateChannel.getUpdateChannel(name = updateChannelName)

            if (this.whatsNewSeen) {
                val packageManager = context.packageManager
                val packageInfo = packageManager.getPackageInfo(context.packageName, 0)
                val versionName = 'v' + packageInfo.versionName!!
                if (this.whatsNewVersionSeen != versionName) {
                    this@SatunesSettings.unSeeWhatsNew(context = context)
                }
            }
        }.first() //Without .first() settings are not loaded correctly
    }

    suspend fun seeWhatsNew(context: Context) {
        context.dataStore.edit { preferences: MutablePreferences ->
            this.whatsNewSeen = true
            preferences[WHATS_NEW_SEEN_KEY] = true
            val packageManager = context.packageManager
            val packageInfo = packageManager.getPackageInfo(context.packageName, 0)
            val versionName = 'v' + packageInfo.versionName!!
            preferences[WHATS_NEW_VERSION_SEEN_KEY] = versionName
            whatsNewVersionSeen = versionName
        }
    }

    suspend fun unSeeWhatsNew(context: Context) {
        context.dataStore.edit { preferences: MutablePreferences ->
            this.whatsNewSeen = false
            preferences[WHATS_NEW_SEEN_KEY] = false
        }
    }

    suspend fun switchLogsActivation(context: Context) {
        context.dataStore.edit { preferences: MutablePreferences ->
            this.logsActivation.value = !this.logsActivation.value
            preferences[LOGS_ACTIVATION_KEY] = this.logsActivation.value
        }
    }

    suspend fun reset(context: Context) {
        context.dataStore.edit { preferences: MutablePreferences ->
            this.logsActivation.value = DEFAULT_LOGS_ACTIVATION
            preferences[LOGS_ACTIVATION_KEY] = this.logsActivation.value
            this.whatsNewSeen = DEFAULT_WHATS_NEW_SEEN
            preferences[WHATS_NEW_SEEN_KEY] = DEFAULT_WHATS_NEW_SEEN
            this.whatsNewVersionSeen = DEFAULT_WHATS_NEW_VERSION_SEEN
            preferences[WHATS_NEW_VERSION_SEEN_KEY] = this.whatsNewVersionSeen
            this.includeExcludeSeen = DEFAULT_INCLUDE_EXCLUDE_SEEN
            preferences[INCLUDE_EXCLUDE_SEEN_KEY] = DEFAULT_INCLUDE_EXCLUDE_SEEN
            this.updateChannel.value = DEFAULT_UPDATE_CHANNEL
            preferences[UPDATE_CHANNEL_KEY] = DEFAULT_UPDATE_CHANNEL.name
        }
    }

    suspend fun seeIncludeExcludeInfo(context: Context) {
        context.dataStore.edit { preferences: MutablePreferences ->
            this.includeExcludeSeen = true
            preferences[INCLUDE_EXCLUDE_SEEN_KEY] = true
        }
    }

    suspend fun unSeeIncludeExcludeInfo(context: Context) {
        context.dataStore.edit { preferences: MutablePreferences ->
            this.includeExcludeSeen = false
            preferences[INCLUDE_EXCLUDE_SEEN_KEY] = false
        }
    }

    suspend fun selectUpdateChannel(context: Context, channel: UpdateChannel) {
        context.dataStore.edit { preferences: MutablePreferences ->
            this.updateChannel.value = channel
            preferences[UPDATE_CHANNEL_KEY] = this.updateChannel.value.name
        }
    }
}