import 'package:flutter/material.dart';
import 'package:photos/core/event_bus.dart';
import 'package:photos/events/collection_meta_event.dart';
import 'package:photos/events/collection_updated_event.dart';
import 'package:photos/events/files_updated_event.dart';
import "package:photos/generated/l10n.dart";
import "package:photos/models/file/extensions/file_props.dart";
import "package:photos/models/file/file.dart";
import 'package:photos/models/file_load_result.dart';
import 'package:photos/models/gallery_type.dart';
import 'package:photos/models/selected_files.dart';
import "package:photos/services/search_service.dart";
import 'package:photos/ui/viewer/actions/file_selection_overlay_bar.dart';
import "package:photos/ui/viewer/gallery/component/group/type.dart";
import 'package:photos/ui/viewer/gallery/gallery.dart';
import "package:photos/ui/viewer/gallery/state/boundary_reporter_mixin.dart";
import "package:photos/ui/viewer/gallery/state/gallery_boundaries_provider.dart";
import "package:photos/ui/viewer/gallery/state/gallery_files_inherited_widget.dart";
import "package:photos/ui/viewer/gallery/state/selection_state.dart";

class LargeFilesPagePage extends StatefulWidget {
  final String tagPrefix;
  final GalleryType appBarType;
  final GalleryType overlayType;
  final _selectedFiles = SelectedFiles();
  static const int minLargeFileSize = 50 * 1024 * 1024;

  LargeFilesPagePage({
    this.tagPrefix = "Uncategorized_page",
    this.appBarType = GalleryType.homepage,
    this.overlayType = GalleryType.homepage,
    super.key,
  });

  @override
  State<LargeFilesPagePage> createState() => _LargeFilesPagePageState();
}

class _LargeFilesPagePageState extends State<LargeFilesPagePage> {
  @override
  Widget build(BuildContext context) {
    final gallery = Gallery(
      asyncLoader: (creationStartTime, creationEndTime, {limit, asc}) async {
        final List<EnteFile> allFiles =
            await SearchService.instance.getAllFilesForSearch();
        final Set<int> alreadyTracked = <int>{};

        final filesWithSize = <EnteFile>[];
        for (final file in allFiles) {
          if (file.isOwner &&
              file.isUploaded &&
              file.fileSize != null &&
              file.fileSize! > LargeFilesPagePage.minLargeFileSize) {
            if (!alreadyTracked.contains(file.uploadedFileID!)) {
              filesWithSize.add(file);
              alreadyTracked.add(file.uploadedFileID!);
            }
          }
        }
        // sort by file size descending
        filesWithSize.sort((a, b) => b.fileSize!.compareTo(a.fileSize!));
        final FileLoadResult result = FileLoadResult(filesWithSize, false);
        return result;
      },
      reloadEvent: Bus.instance.on<CollectionUpdatedEvent>(),
      removalEventTypes: const {
        EventType.deletedFromRemote,
        EventType.deletedFromEverywhere,
        EventType.hide,
      },
      forceReloadEvents: [
        Bus.instance.on<CollectionMetaEvent>(),
      ],
      tagPrefix: widget.tagPrefix,
      selectedFiles: widget._selectedFiles,
      sortAsyncFn: () => false,
      groupType: GroupType.size,
      initialFiles: null,
      albumName: AppLocalizations.of(context).viewLargeFiles,
    );
    return GalleryBoundariesProvider(
      child: GalleryFilesState(
        child: Scaffold(
          appBar: const PreferredSize(
            preferredSize: Size.fromHeight(50.0),
            child: _LargeFilesAppBar(),
          ),
          body: SelectionState(
            selectedFiles: widget._selectedFiles,
            child: Stack(
              alignment: Alignment.bottomCenter,
              children: [
                gallery,
                FileSelectionOverlayBar(
                  widget.overlayType,
                  widget._selectedFiles,
                ),
              ],
            ),
          ),
        ),
      ),
    );
  }
}

class _LargeFilesAppBar extends StatefulWidget {
  const _LargeFilesAppBar();

  @override
  State<_LargeFilesAppBar> createState() => _LargeFilesAppBarState();
}

class _LargeFilesAppBarState extends State<_LargeFilesAppBar>
    with BoundaryReporter {
  @override
  Widget build(BuildContext context) {
    return boundaryWidget(
      position: BoundaryPosition.top,
      child: AppBar(
        elevation: 0,
        centerTitle: false,
        title: Text(
          AppLocalizations.of(context).viewLargeFiles,
          style:
              Theme.of(context).textTheme.headlineSmall!.copyWith(fontSize: 16),
          maxLines: 2,
          overflow: TextOverflow.ellipsis,
        ),
      ),
    );
  }
}
