import 'dart:ui';

import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';

import '../../theme.dart';

/// Applies a theme to descendant [SfDataPager] widgets.
class SfDataPagerTheme extends InheritedTheme {
  /// Applies the given theme [data] to [child].
  const SfDataPagerTheme({Key? key, required this.data, required this.child})
    : super(key: key, child: child);

  /// Specifies the color and typography values for descendant [SfDataPager]
  /// widgets.
  final SfDataPagerThemeData data;

  /// The widget below this widget in the tree.
  @override
  final Widget child;

  /// The data from the closest [SfDataPagerTheme]
  /// instance that encloses the given context.
  ///
  /// Defaults to [SfThemeData.dataPagerThemeData] if there
  /// is no [SfDataPagerTheme] in the given build context.
  static SfDataPagerThemeData? of(BuildContext context) {
    final SfDataPagerTheme? sfDataPagerTheme =
        context.dependOnInheritedWidgetOfExactType<SfDataPagerTheme>();
    return sfDataPagerTheme?.data ?? SfTheme.of(context).dataPagerThemeData;
  }

  @override
  bool updateShouldNotify(SfDataPagerTheme oldWidget) => data != oldWidget.data;

  @override
  Widget wrap(BuildContext context, Widget child) {
    final SfDataPagerTheme? ancestorTheme =
        context.findAncestorWidgetOfExactType<SfDataPagerTheme>();
    return identical(this, ancestorTheme)
        ? child
        : SfDataPagerTheme(data: data, child: child);
  }
}

/// Holds the color and typography values for a [SfDataPagerTheme]. Use
///  this class to configure a [SfDataPagerTheme] widget
///
/// To obtain the current theme, use [SfDataPagerTheme.of].
@immutable
class SfDataPagerThemeData with Diagnosticable {
  /// Create a [SfDataPagerThemeData] that's used to configure a
  /// [SfDataPagerTheme].
  const SfDataPagerThemeData({
    this.backgroundColor,
    this.itemColor,
    this.itemTextStyle,
    this.selectedItemColor,
    this.selectedItemTextStyle,
    this.disabledItemColor,
    this.disabledItemTextStyle,
    this.itemBorderColor,
    this.itemBorderWidth,
    this.itemBorderRadius,
    this.dropdownButtonBorderColor,
  });

  /// Create a [SfDataPagerThemeData] that's used to configure a
  /// [SfDataPagerTheme].
  factory SfDataPagerThemeData.raw({
    Brightness? brightness,
    Color? backgroundColor,
    Color? itemColor,
    TextStyle? itemTextStyle,
    Color? selectedItemColor,
    TextStyle? selectedItemTextStyle,
    Color? disabledItemColor,
    TextStyle? disabledItemTextStyle,
    Color? itemBorderColor,
    double? itemBorderWidth,
    BorderRadiusGeometry? itemBorderRadius,
    Color? dropdownButtonBorderColor,
  }) {
    brightness = brightness ?? Brightness.light;
    return SfDataPagerThemeData(
      backgroundColor: backgroundColor,
      itemColor: itemColor,
      itemTextStyle: itemTextStyle,
      selectedItemColor: selectedItemColor,
      selectedItemTextStyle: selectedItemTextStyle,
      disabledItemColor: disabledItemColor,
      disabledItemTextStyle: disabledItemTextStyle,
      itemBorderColor: itemBorderColor,
      itemBorderWidth: itemBorderWidth,
      itemBorderRadius: itemBorderRadius,
      dropdownButtonBorderColor: dropdownButtonBorderColor,
    );
  }

  /// The color of the page Items
  final Color? itemColor;

  /// The color of the data pager background
  final Color? backgroundColor;

  /// The style of the text of page Items
  final TextStyle? itemTextStyle;

  /// The color of the page Items which are disabled.
  final Color? disabledItemColor;

  /// The style of the text of page items which are disabled.
  final TextStyle? disabledItemTextStyle;

  /// The color of the currently selected page item.
  final Color? selectedItemColor;

  /// The style of the text of currently selected page Item.
  final TextStyle? selectedItemTextStyle;

  /// The color of the border in page Item.
  final Color? itemBorderColor;

  /// The width of the border in page item.
  final double? itemBorderWidth;

  /// If non null, the corners of the page item are rounded by
  /// this [ItemBorderRadius].
  ///
  /// Applies only to boxes with rectangular shapes;
  /// see also:
  ///
  /// [BoxDecoration.borderRadius]
  final BorderRadiusGeometry? itemBorderRadius;

  ///The border color of the rowsPerPage dropdown button.
  final Color? dropdownButtonBorderColor;

  /// Creates a copy of this theme but with the given
  /// fields replaced with the new values.
  SfDataPagerThemeData copyWith({
    Brightness? brightness,
    Color? backgroundColor,
    Color? itemColor,
    TextStyle? itemTextStyle,
    Color? selectedItemColor,
    TextStyle? selectedItemTextStyle,
    Color? disabledItemColor,
    TextStyle? disabledItemTextStyle,
    Color? itemBorderColor,
    double? itemBorderWidth,
    BorderRadiusGeometry? itemBorderRadius,
    Color? dropdownButtonBorderColor,
  }) {
    return SfDataPagerThemeData.raw(
      brightness: brightness,
      backgroundColor: backgroundColor ?? this.backgroundColor,
      itemColor: itemColor ?? this.itemColor,
      itemTextStyle: itemTextStyle ?? this.itemTextStyle,
      selectedItemColor: selectedItemColor ?? this.selectedItemColor,
      selectedItemTextStyle:
          selectedItemTextStyle ?? this.selectedItemTextStyle,
      disabledItemColor: disabledItemColor ?? this.disabledItemColor,
      disabledItemTextStyle:
          disabledItemTextStyle ?? this.disabledItemTextStyle,
      itemBorderColor: itemBorderColor ?? this.itemBorderColor,
      itemBorderWidth: itemBorderWidth ?? this.itemBorderWidth,
      itemBorderRadius: itemBorderRadius ?? this.itemBorderRadius,
      dropdownButtonBorderColor:
          dropdownButtonBorderColor ?? this.dropdownButtonBorderColor,
    );
  }

  /// Linearly interpolate between two themes.
  static SfDataPagerThemeData? lerp(
    SfDataPagerThemeData? a,
    SfDataPagerThemeData? b,
    double t,
  ) {
    if (a == null && b == null) {
      return null;
    }
    return SfDataPagerThemeData(
      backgroundColor: Color.lerp(a!.backgroundColor, b!.backgroundColor, t),
      itemColor: Color.lerp(a.itemColor, b.itemColor, t),
      itemTextStyle: TextStyle.lerp(a.itemTextStyle, b.itemTextStyle, t),
      selectedItemColor: Color.lerp(
        a.selectedItemColor,
        b.selectedItemColor,
        t,
      ),
      selectedItemTextStyle: TextStyle.lerp(
        a.selectedItemTextStyle,
        b.selectedItemTextStyle,
        t,
      ),
      disabledItemColor: Color.lerp(
        a.disabledItemColor,
        b.disabledItemColor,
        t,
      ),
      disabledItemTextStyle: TextStyle.lerp(
        a.disabledItemTextStyle,
        b.disabledItemTextStyle,
        t,
      ),
      itemBorderColor: Color.lerp(a.itemBorderColor, b.itemBorderColor, t),
      itemBorderWidth: lerpDouble(a.itemBorderWidth, b.itemBorderWidth, t),
      itemBorderRadius: BorderRadiusGeometry.lerp(
        a.itemBorderRadius,
        b.itemBorderRadius,
        t,
      ),
      dropdownButtonBorderColor: Color.lerp(
        a.dropdownButtonBorderColor,
        b.dropdownButtonBorderColor,
        t,
      ),
    );
  }

  @override
  bool operator ==(Object other) {
    if (identical(this, other)) {
      return true;
    }
    if (other.runtimeType != runtimeType) {
      return false;
    }

    return other is SfDataPagerThemeData &&
        other.itemColor == itemColor &&
        other.backgroundColor == backgroundColor &&
        other.itemTextStyle == itemTextStyle &&
        other.selectedItemColor == selectedItemColor &&
        other.selectedItemTextStyle == selectedItemTextStyle &&
        other.disabledItemColor == disabledItemColor &&
        other.disabledItemTextStyle == disabledItemTextStyle &&
        other.itemBorderColor == itemBorderColor &&
        other.itemBorderWidth == itemBorderWidth &&
        other.itemBorderRadius == itemBorderRadius &&
        other.dropdownButtonBorderColor == dropdownButtonBorderColor;
  }

  @override
  int get hashCode {
    final List<Object?> values = <Object?>[
      itemColor,
      backgroundColor,
      itemTextStyle,
      selectedItemColor,
      selectedItemTextStyle,
      disabledItemColor,
      disabledItemTextStyle,
      itemBorderColor,
      itemBorderWidth,
      itemBorderRadius,
      dropdownButtonBorderColor,
    ];
    return Object.hashAll(values);
  }

  @override
  void debugFillProperties(DiagnosticPropertiesBuilder properties) {
    super.debugFillProperties(properties);
    const SfDataPagerThemeData defaultData = SfDataPagerThemeData();
    properties.add(
      ColorProperty(
        'backgroundColor',
        backgroundColor,
        defaultValue: defaultData.backgroundColor,
      ),
    );
    properties.add(
      ColorProperty(
        'itemColor',
        itemColor,
        defaultValue: defaultData.itemColor,
      ),
    );
    properties.add(
      ColorProperty(
        'selectedItemColor',
        selectedItemColor,
        defaultValue: defaultData.selectedItemColor,
      ),
    );
    properties.add(
      ColorProperty(
        'disabledItemColor',
        disabledItemColor,
        defaultValue: defaultData.disabledItemColor,
      ),
    );
    properties.add(
      ColorProperty(
        'itemBorderColor',
        itemBorderColor,
        defaultValue: defaultData.itemBorderColor,
      ),
    );
    properties.add(
      DiagnosticsProperty<TextStyle>(
        'itemTextStyle',
        itemTextStyle,
        defaultValue: defaultData.itemTextStyle,
      ),
    );
    properties.add(
      DiagnosticsProperty<TextStyle>(
        'selectedItemTextStyle',
        selectedItemTextStyle,
        defaultValue: defaultData.selectedItemTextStyle,
      ),
    );
    properties.add(
      DiagnosticsProperty<TextStyle>(
        'disabledItemTextStyle',
        disabledItemTextStyle,
        defaultValue: defaultData.disabledItemTextStyle,
      ),
    );
    properties.add(
      DoubleProperty(
        'itemBorderWidth',
        itemBorderWidth,
        defaultValue: defaultData.itemBorderWidth,
      ),
    );
    properties.add(
      DiagnosticsProperty<BorderRadiusGeometry>(
        'itemBorderRadius',
        itemBorderRadius,
        defaultValue: defaultData.itemBorderRadius,
      ),
    );
    properties.add(
      ColorProperty(
        'dropdownButtonBorderColor',
        dropdownButtonBorderColor,
        defaultValue: defaultData.dropdownButtonBorderColor,
      ),
    );
  }
}
